#pragma once

#include "sender.h"
#include "composer.h"
#include <typed_log/typed_log.h>
#include <ymod_smtpclient/call.h>
#include <yplatform/find.h>
#include <yplatform/module.h>

namespace botserver::mail {

struct settings
{
    string smtp_host;
    unsigned short smtp_port;
    string sender_email;

    void parse_ptree(yplatform::ptree conf)
    {
        smtp_host = conf.get<string>("smtp.host");
        smtp_port = conf.get<unsigned short>("smtp.port");
        sender_email = conf.get<string>("sender");
    }
};

inline settings make_settings(yplatform::ptree conf)
{
    settings ret;
    ret.parse_ptree(conf);
    return ret;
}

template <typename SmtpClient>
struct module_impl
    : sender
    , yplatform::module
{
    using smtp_client = SmtpClient;
    using smtp_client_ptr = shared_ptr<SmtpClient>;
    using smtp_request = ymod_smtpclient::Request;

    settings settings;
    smtp_client_ptr smtp;
    composer composer;

    module_impl(struct settings st) : settings(st), smtp(find_module<smtp_client>("smtp_client"))
    {
    }

    module_impl(yplatform::ptree conf) : module_impl(make_settings(conf))
    {
    }

    future<void> send(task_context_ptr ctx, mail_message_ptr message) override
    {
        promise<void> prom;
        smtp_request req;
        req.address.host = settings.smtp_host;
        req.address.port = settings.smtp_port;
        req.mailfrom.email = settings.sender_email;
        req.rcpts.emplace_back(message->to_email);
        if (message->attachments.empty())
        {
            req.message = composer.compose_text_message(message, settings.sender_email);
        }
        else
        {
            req.message = composer.compose_message_with_attachments(message, settings.sender_email);
        }
        time_point sent_start_ts = clock::now();
        smtp->asyncRun(
            ctx, req, [prom, ctx, message, sent_start_ts](error_code err, auto resp) mutable {
                string smtp_response = resp.session ? resp.session->data : "";
                if (err)
                {
                    typed::log_mail_message_sent(
                        ctx, "error", err.message(), smtp_response, sent_start_ts, message->from);
                    prom.set_exception(runtime_error("smtp error: " + err.message()));
                    return;
                }
                else
                {
                    typed::log_mail_message_sent(
                        ctx, "success", "", smtp_response, sent_start_ts, message->from);
                    prom.set();
                }
            });
        return prom;
    }
};

using module = module_impl<ymod_smtpclient::Call>;

}