#pragma once

#include "context.h"
#include "validation.h"
#include <i18n/i18n.h>
#include <common/types.h>

namespace botserver::message_processor {

struct code
{
    void operator()(context_ptr context)
    {
        string code = context->command.args["code"];
        if (code.empty())
        {
            reply_missing_code(context);
            return;
        }
        if (!valid_otp(code))
        {
            reply_invalid_code(context);
            return;
        }
        auto check_result = check_otp(context, code);
        if (!check_result.valid)
        {
            report_check_failed(context);
            reply_invalid_code(context);
            return;
        }
        report_check_success(context);
        add_link(context, check_result.mail_account);
        reply_bind_successfully(context);
    }

    db::otp_check_result check_otp(context_ptr context, string code)
    {
        return context->otp->check_code(context, context->botpeer, code);
    }

    void report_check_failed(context_ptr context)
    {
        context->custom_log_data["check_code"] = "fail";
    }

    void report_check_success(context_ptr context)
    {
        context->custom_log_data["check_code"] = "success";
    }

    void reply_missing_code(context_ptr context)
    {
        context->gate->send_message(context, context->botpeer, i18n::missing_code(context->lang()))
            .get();
    }

    void reply_invalid_code(context_ptr context)
    {
        context->gate
            ->send_message(context, context->botpeer, i18n::invalid_code_number(context->lang()))
            .get();
    }

    void add_link(context_ptr context, mail_account account)
    {
        link link = { .botpeer = context->botpeer, .mail_account = account };
        context->links->add(context, link);
    }

    void reply_bind_successfully(context_ptr context)
    {
        context->gate
            ->send_message(
                context, context->botpeer, i18n::email_binded_successfully(context->lang()))
            .get();
    }
};

}