#pragma once

#include "validation.h"
#include "context.h"
#include <i18n/i18n.h>
#include <common/types.h>

namespace botserver::message_processor {

struct email
{
    void operator()(context_ptr context)
    {
        string email_arg = context->command.args["email"];
        if (email_arg.empty())
        {
            reply_missing_email(context);
            return;
        }
        if (!valid_email(email_arg))
        {
            reply_invalid_email(context);
            return;
        }
        auto account = get_mail_account(context, email_arg);
        if (!account)
        {
            report_no_such_account(context, email_arg);
            reply_code_sent(context, email_arg);
            return;
        }
        report_user_data(context, *account);
        if (!check_rate_limit(context, context->botpeer, account->uid))
        {
            report_rate_limit_exceeded(context);
            reply_otp_limit_exceeded(context);
            return;
        }
        string otp_code = gen_otp(context, *account);
        send_otp_to_email(context, account->email, otp_code);
        reply_code_sent(context, account->email);
    }

    optional<mail_account> get_mail_account(context_ptr context, string email)
    {
        return context->account_provider->get_mail_account(context, context->botpeer, email).get();
    }

    void report_no_such_account(context_ptr context, string email)
    {
        context->custom_log_data["uid"] = "";
        context->custom_log_data["email"] = email;
    }

    void report_user_data(context_ptr context, mail_account account)
    {
        context->custom_log_data["uid"] = account.uid;
        context->custom_log_data["email"] = account.email;
    }

    void reply_code_sent(context_ptr context, string email)
    {
        context->gate
            ->send_message(
                context, context->botpeer, i18n::send_code_from_email(context->lang(), email))
            .get();
    }

    void reply_missing_email(context_ptr context)
    {
        context->gate->send_message(context, context->botpeer, i18n::missing_email(context->lang()))
            .get();
    }

    void reply_invalid_email(context_ptr context)
    {
        context->gate->send_message(context, context->botpeer, i18n::invalid_email(context->lang()))
            .get();
    }

    bool check_rate_limit(context_ptr context, botpeer botpeer, string uid)
    {
        return context->otp_limiter->acquire(botpeer, uid);
    }

    void reply_otp_limit_exceeded(context_ptr context)
    {
        context->gate->send_message(
            context, context->botpeer, i18n::otp_rate_limit_exceeded(context->lang()));
    }

    string gen_otp(context_ptr context, mail_account account)
    {
        return context->otp->gen_code(context, context->botpeer, account);
    }

    void send_otp_to_email(context_ptr context, string email, string otp_code)
    {
        auto message = make_mail_message(context, email, otp_code);
        context->mail_sender->send(context, message).get();
    }

    mail_message_ptr make_mail_message(context_ptr context, string email, string otp_code)
    {
        auto message = make_shared<mail_message>();
        message->subject = i18n::your_otp_code_for_bot(context->lang(), otp_code);
        message->from = context->botpeer;
        message->to_email = email;
        message->text = i18n::otp_code_is(context->lang(), context->botpeer.username, otp_code);
        return message;
    }

    void report_rate_limit_exceeded(context_ptr context)
    {
        context->custom_log_data["skip_reason"] = "rate limit exceeded";
    }
};

}