#pragma once

#include <message_processor/context.h>
#include <db/interface.h>
#include <i18n/i18n.h>
#include <common/types.h>
#include <typed_log/typed_log.h>

namespace botserver::message_processor {

struct forward
{
    void operator()(context_ptr context)
    {
        auto link = context->links->lookup(context, context->botpeer);
        if (!link)
        {
            report_skip_processing(context, "no link");
            context->gate
                ->send_message(context, context->botpeer, i18n::bind_email_address(context->lang()))
                .get();
            return;
        }

        report_dst_email(context, link->mail_account);
        report_attachments(context, context->message);
        if (max_message_size_exceeded(context))
        {
            report_skip_processing(context, "message too big");
            reply_message_too_big(context);
            return;
        }
        auto attachments = download_attachments(context, context->message->attachments);
        auto mail_message = make_mail_message(context, *link, attachments);
        context->mail_sender->send(context, mail_message).get();
    }

    bool max_message_size_exceeded(context_ptr context)
    {
        if (!context->settings.max_message_size) return false;
        size_t total_size =
            context->message->text.size() + total_attachments_size(context->message);
        return total_size > context->settings.max_message_size;
    }

    void report_skip_processing(context_ptr context, string reason)
    {
        context->custom_log_data["skip_reason"] = reason;
    }

    void reply_message_too_big(context_ptr context)
    {
        context->gate
            ->send_message(context, context->botpeer, i18n::message_too_big(context->lang()))
            .get();
    }

    vector<mail_attachment> download_attachments(
        context_ptr context,
        vector<attachment_meta> attachments_meta)
    {
        vector<mail_attachment> attachments;
        for (auto&& attachment_meta : attachments_meta)
        {
            attachments.push_back(download_attachment(context, attachment_meta));
        }
        return attachments;
    }

    mail_attachment download_attachment(context_ptr context, attachment_meta attachment_meta)
    {
        return {
            .name = attachment_meta.file_name,
            .mime_type = attachment_meta.mime_type,
            .content =
                context->gate->download_file(context, context->botpeer, attachment_meta.id).get(),
        };
    }

    mail_message_ptr make_mail_message(
        context_ptr context,
        link link,
        vector<mail_attachment> attachments)
    {
        auto message = context->message;
        auto mail_message = std::make_shared<botserver::mail_message>();

        if (message->forwarded_from)
        {
            mail_message->subject = i18n::forwarded_from_user_on_messenger(
                context->lang(), *message->forwarded_from, context->botpeer.platform);
        }
        else
        {
            mail_message->subject =
                i18n::forwarded_from_messenger(context->lang(), context->botpeer.platform);
        }

        if (!message->forwarded_from || !message->received_date)
        {
            mail_message->text = message->text;
        }
        else
        {
            mail_message->text = i18n::received_from_user(
                context->lang(), message->text, *message->forwarded_from, message->received_date);
        }

        mail_message->from = context->botpeer;
        mail_message->to_email = link.mail_account.email;
        mail_message->attachments = attachments;

        return mail_message;
    }

    void report_attachments(context_ptr context, gate_message_ptr message)
    {
        context->custom_log_data["attachments_count"] = std::to_string(message->attachments.size());
        context->custom_log_data["attachments_total_size"] =
            std::to_string(total_attachments_size(message));
    }

    void report_dst_email(context_ptr context, mail_account acc)
    {
        context->custom_log_data["dst_email"] = acc.email;
    }

    size_t total_attachments_size(gate_message_ptr message)
    {
        size_t total_size = 0;
        for (auto&& att : message->attachments)
        {
            total_size += att.size;
        }
        return total_size;
    }
};

}