#pragma once

#include "parser.h"
#include "forward.h"
#include "start.h"
#include "email.h"
#include "code.h"
#include "ping.h"
#include "settings.h"
#include <typed_log/typed_log.h>
#include <common/gate_message.h>
#include <common/types.h>
#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/find.h>
#include <yplatform/module.h>
#include <yplatform/reactor.h>
#include <yplatform/util/weak_bind.h>

namespace botserver::message_processor {

struct traits
{
    using forward = message_processor::forward;
    using ping = message_processor::ping;
    using start = message_processor::start;
    using email = message_processor::email;
    using code = message_processor::code;
};

template <typename Traits>
struct module_impl : yplatform::module
{
    using this_type = module_impl<Traits>;
    using forward_command = typename Traits::forward;
    using ping_command = typename Traits::ping;
    using start_command = typename Traits::start;
    using email_command = typename Traits::email;
    using code_command = typename Traits::code;

    yplatform::reactor_ptr reactor;
    settings settings;
    ymod_ratecontroller::rate_controller_ptr rc, attachments_rc;
    messenger::gate_ptr gate;
    mail::sender_ptr mail_sender;
    db::links_ptr links;
    db::otp_ptr otp;
    auth::account_provider_ptr account_provider;
    otp_rate_limiter_ptr otp_limiter;

    module_impl(yplatform::reactor& reactor, struct settings st)
        : reactor(yplatform::reactor::make_not_owning_copy(reactor)), settings(st)
    {
        auto rate_controller_module = yplatform::find<ymod_ratecontroller::rate_controller_module>(
            *reactor.io(), "rate_controller");
        rc = rate_controller_module->get_controller("messages");
        attachments_rc = rate_controller_module->get_controller("messages_with_attachments");
        gate = yplatform::find<messenger::gate, std::shared_ptr>(*reactor.io(), st.gate_module);
        mail_sender = find_module<mail::sender>(*reactor.io(), "mail_sender");
        links = find_module<db::links>(*reactor.io(), "botdb");
        otp = find_module<db::otp>(*reactor.io(), "botdb");
        account_provider = find_module<auth::account_provider>(*reactor.io(), "auth");
        otp_limiter =
            make_shared<otp_rate_limiter>(settings.otp_botpeer_limit, settings.otp_uid_limit);
        ensure_reactor_configuration();
    }

    module_impl(yplatform::reactor& reactor, yplatform::ptree conf)
        : module_impl(reactor, make_settings(conf))
    {
    }

    void ensure_reactor_configuration()
    {
        if (reactor->size() != 1)
        {
            throw runtime_error("message_processor requires a multi-thread reactor - set "
                                "pool_count=1 and io_threads=N");
        }
        if ((*reactor)[0]->size() != rc->max_concurrency() + attachments_rc->max_concurrency())
        {
            throw runtime_error("io threads in message_processor reactor should be equal to "
                                "total concurrency of corresponding rate controllers");
        }
    }

    void init()
    {
        gate->set_message_handler(yplatform::weak_bind(
            &this_type::on_message_received, shared_from(this), ph::_1, ph::_2, ph::_3));
    }

    future<void> on_message_received(
        task_context_ptr task_ctx,
        botpeer botpeer,
        gate_message_ptr message)
    {
        promise<void> prom;
        auto context = make_context(task_ctx, message, botpeer);
        get_rc(message)->post(reactor->io()->wrap(
            [this, capture_self, context, prom](auto err, auto on_complete) mutable {
                context->start_process_ts = clock::now();
                if (err)
                {
                    report_message_processed(context, "error", err.message());
                    prom.set_exception(runtime_error("rate controller error: " + err.message()));
                    return on_complete();
                }
                try
                {
                    process_message(context);
                    report_message_processed(context, "success");
                    prom.set();
                }
                catch (const exception& e)
                {
                    report_message_processed(context, "error", e.what());
                    prom.set_current_exception();
                }
                on_complete();
            }));
        return prom;
    }

    auto make_context(task_context_ptr task_ctx, gate_message_ptr message, botpeer botpeer)
    {
        auto context = boost::make_shared<struct context>(*task_ctx);
        context->message = message;
        context->botpeer = botpeer;
        context->settings = settings;
        context->gate = gate;
        context->mail_sender = mail_sender;
        context->links = links;
        context->otp = otp;
        context->account_provider = account_provider;
        context->otp_limiter = otp_limiter;
        return context;
    }

    auto get_rc(gate_message_ptr message)
    {
        return message->attachments.size() ? attachments_rc : rc;
    }

    void report_message_processed(context_ptr context, string status, string reason = "")
    {
        typed::log_message_processed(
            context,
            status,
            reason,
            context->botpeer,
            std::to_string(
                static_cast<int>(context->command.name)), // XXX log command with magic enum lib
            context->create_ts,
            context->start_process_ts,
            context->custom_log_data);
    }

    void process_message(context_ptr context)
    {
        context->command = parser()(context->message);
        auto processor = make_command_processor(context->command);
        processor(context);
    }

    function<void(context_ptr)> make_command_processor(command command)
    {
        switch (command.name)
        {
        case command_name::start:
            return start_command{};
        case command_name::ping:
            return ping_command{};
        case command_name::email:
            return email_command{};
        case command_name::code:
            return code_command{};
        case command_name::debug:
            throw runtime_error("not implemented command");
        case command_name::forward:
            return forward_command{};
        case command_name::unknown:
            throw runtime_error("unknown command");
        }
        throw runtime_error("unknown command");
    }
};

using module = module_impl<traits>;

}