#pragma once

#include <common/json.h>
#include <common/types.h>
#include <ymod_httpclient/request.h>
#include <ymod_httpclient/call.h>
#include <ymod_httpclient/url_encode.h>
#include <yplatform/json.h>

namespace botserver::messenger::telegram {

struct file_info
{
    string id;
    string unique_id;
    size_t size;
    string path;
};

template <typename HttpClient>
class api
{
    using http_client_ptr = shared_ptr<HttpClient>;

    http_client_ptr client;
    string token;

public:
    api(http_client_ptr client, string token) : client(client), token(token)
    {
    }

    future<void> send_message(
        task_context_ptr ctx,
        string chat_id,
        string text,
        string parse_mode = "")
    {
        promise<void> res;
        auto request = yhttp::request::POST(
            "/bot" + token + "/sendMessage",
            yhttp::form_encode(
                { { "chat_id", chat_id }, { "text", text }, { "parse_mode", parse_mode } }));
        auto process_result = [](auto promise, auto /*result*/) { promise.set(); };
        client->async_run(
            ctx, request, [this, res, process_result](auto ec, yhttp::response response) {
                this->handle_api_response(ec, response, res, process_result);
            });
        return res;
    }

    future<file_info> get_file_info(task_context_ptr ctx, string file_id)
    {
        promise<file_info> res;
        auto request = yhttp::request::POST(
            "/bot" + token + "/getFile", yhttp::form_encode({ { "file_id", file_id } }));
        auto process_result = [](auto promise, json_value result) {
            promise.set({ .id = get_required_field<string>(result, "file_id"),
                          .unique_id = get_required_field<string>(result, "file_unique_id"),
                          .size = get_optional_field<uint64_t>(result, "file_size"),
                          .path = get_optional_field<string>(result, "file_path") });
        };
        client->async_run(
            ctx, request, [this, res, process_result](auto ec, yhttp::response response) {
                this->handle_api_response(ec, response, res, process_result);
            });
        return res;
    }

    future<string_ptr> download_file(task_context_ptr ctx, string file_path)
    {
        promise<string_ptr> res;
        auto request = yhttp::request::GET("/file/bot" + token + "/" + file_path);
        client->async_run(ctx, request, [res](auto ec, yhttp::response response) mutable {
            if (ec)
                return res.set_exception(
                    runtime_error("telegram api request error: " + ec.message()));
            if (response.status / 100 != 2)
                return res.set_exception(runtime_error(
                    "telegram api bad http status, code=" + std::to_string(response.status) +
                    ", body=" + response.body));
            res.set(make_shared<string>(std::move(response.body)));
        });
        return res;
    }

private:
    template <typename Promise, typename ProcessFunc>
    void handle_api_response(
        error_code ec,
        yhttp::response response,
        Promise prom,
        ProcessFunc process)
    {
        if (ec)
            return prom.set_exception(runtime_error("telegram api request error: " + ec.message()));
        if (response.status / 100 != 2)
            return prom.set_exception(runtime_error(
                "telegram api bad http status, code=" + std::to_string(response.status) +
                ", body=" + response.body));
        try
        {
            json_value json;
            if (auto error = json.parse(response.body))
            {
                throw runtime_error("telegram api parse error: " + *error);
            }

            if (has_error(json))
            {
                throw runtime_error(
                    "telegram api error: " + get_error_description(json, response.body));
            }

            process(prom, json["result"]);
        }
        catch (...)
        {
            prom.set_current_exception();
        }
    }

    bool has_error(const json_value& json)
    {
        return !get_optional_field<bool>(json, "ok", false);
    }

    string get_error_description(const json_value& json, const string& fallback)
    {
        auto description = get_optional_field<string>(json, "description");
        if (description.empty())
        {
            description = fallback;
        }
        return description;
    }
};

}
