#pragma once

#include "api.h"
#include "settings.h"
#include "webhook_update_receiver.h"
#include <common/errors.h>
#include <common/json.h>
#include <common/types.h>
#include <messenger/gate.h>
#include <typed_log/typed_log.h>
#include <ymod_httpclient/cluster_client.h>
#include <ymod_webserver/server.h>
#include <yplatform/algorithm/lru_cache.h>
#include <yplatform/module.h>
#include <yplatform/util/weak_bind.h>
#include <boost/format.hpp>

namespace botserver::messenger::telegram {

inline settings make_settings(const yplatform::ptree& conf)
{
    struct settings res;
    res.parse_ptree(conf);
    return res;
}

template <typename Webserver, typename HttpClient>
struct module_impl
    : public gate
    , public yplatform::module
{
    using this_type = module_impl<Webserver, HttpClient>;
    using error_counters_cache = yplatform::lru_cache<string, size_t>;

    settings settings;
    gate::message_handler message_handler;
    api<HttpClient> api;
    webhook_update_receiver<Webserver> message_receiver;
    atomic<time_point> processing_disabled_ts = { time_point::min() };

    error_counters_cache error_counters;
    std::mutex counters_mutex;

    module_impl(struct settings settings)
        : settings(settings)
        , api(find_module<HttpClient>("telegram_client"), settings.api_token)
        , message_receiver(find_module<Webserver>("web_server"), settings.bind_path)
        , error_counters(settings.telegram_retries.counters_cache_capacity)
    {
    }

    module_impl(yplatform::ptree conf) : module_impl(make_settings(conf))
    {
    }

    void init()
    {
        message_receiver.set_update_handler(
            yplatform::weak_bind(&this_type::process_update, shared_from(this), ph::_1, ph::_2));
    }

    void start()
    {
        if (!message_handler) throw runtime_error("message handler not set!");
    }

    void set_message_handler(gate::message_handler h) override
    {
        message_handler = h;
    }

    future<void> send_message(task_context_ptr ctx, botpeer peer, string text) override
    {
        return api.send_message(ctx, peer.chat_id, text);
    }

    future<void> send_message(task_context_ptr ctx, botpeer peer, markdown_string text) override
    {
        return api.send_message(ctx, peer.chat_id, text, "MarkdownV2");
    }

    future<void> send_message(task_context_ptr ctx, botpeer peer, html_string text) override
    {
        return api.send_message(ctx, peer.chat_id, text, "HTML");
    }

    future<string_ptr> download_file(task_context_ptr ctx, botpeer peer, string file_id) override
    {
        return api.get_file_info(ctx, file_id)
            .then([ctx, capture_self](auto future) {
                auto file = future.get();
                return self->api.download_file(ctx, file.path);
            })
            .then([ctx, peer, file_id, capture_self](auto future) {
                self->report_download_file(ctx, peer, file_id, future);
                return future.get();
            });
    }

    void disable_processing(task_context_ptr ctx, duration ttl) override
    {
        typed::log_maintenance(
            ctx,
            "disable_processing",
            { { "ttl_sec", std::to_string(duration_cast<seconds>(ttl).count()) } });
        processing_disabled_ts = clock::now() + ttl;
    }

    void report_download_file(
        task_context_ptr ctx,
        botpeer peer,
        string file_id,
        future<string_ptr> future)
    {
        typed::log_download_file(
            ctx,
            future.has_exception() ? "error" : "success",
            get_exception_reason(future),
            peer,
            file_id,
            future.has_exception() ? 0 : future.get()->size());
    }

    future<void> process_update(task_context_ptr ctx, json_value json)
    {
        auto update_id = json["update_id"].to_string();
        if (json.has_member("message"))
        {
            return process_message(ctx, json["message"], update_id);
        }
        else
        {
            // Ignore updates of unknown types
            promise<void> res;
            res.set();
            return res;
        }
    }

    future<void> process_message(task_context_ptr ctx, json_value json, string update_id)
    {
        try
        {
            auto peer = parse_botpeer_from_message(json);
            auto message = parse_gate_message(json);
            if (processing_disabled())
            {
                return send_message(ctx, peer, i18n::service_unavailable(message->lang));
            }
            if (!user_allowed(peer.username))
            {
                // Silently ignore
                return make_future();
            }
            return message_handler(ctx, peer, message)
                .then([update_id, ctx, peer, message, this, capture_self](future<void> future) {
                    if (future.has_exception())
                    {
                        return process_message_handler_error(ctx, future, update_id, peer, message);
                    }
                    return future;
                });
        }
        catch (...)
        {
            promise<void> res;
            res.set_current_exception();
            return res;
        }
    }

    bool processing_disabled()
    {
        return clock::now() < processing_disabled_ts.load();
    }

    bool user_allowed(string username)
    {
        if (settings.blacklist.size() && settings.blacklist.contains(username))
        {
            return false;
        }

        if (settings.whitelist.size() && !settings.whitelist.contains(username))
        {
            return false;
        }
        return true;
    }

    future<void> process_message_handler_error(
        task_context_ptr ctx,
        future<void> res,
        string update_id,
        botpeer peer,
        gate_message_ptr message)
    {
        auto errors = increment_errors_count(update_id);
        if (errors > settings.telegram_retries.limit)
        {
            // If handler cannot process message after retries limit - notify user and ignore error
            return send_message(ctx, peer, i18n::error_occured(message->lang, ctx->uniq_id()));
        }
        return res;
    }

    size_t increment_errors_count(string update_id)
    {
        std::lock_guard lock(counters_mutex);
        auto prev = error_counters.get(update_id);
        auto cur = prev ? *prev + 1 : 1;
        error_counters.put(update_id, cur);
        return cur;
    }

    botpeer parse_botpeer_from_message(json_value json)
    {
        auto user = get_optional_field<json_value>(json, "user");
        auto fallback_username = get_optional_field<string>(user, "username");
        auto chat = get_required_field<json_value>(json, "chat");
        return { .platform = platform_name::telegram,
                 .bot_id = settings.bot_id,
                 .chat_id = get_required_field<string>(chat, "id"),
                 .username = get_optional_field<string>(chat, "username", fallback_username) };
    }

    gate_message_ptr parse_gate_message(json_value json)
    {
        auto caption = get_optional_field<string>(json, "caption");
        return make_shared<gate_message>(
            gate_message{ .lang = parse_language(get_optional_field<json_value>(json, "from")),
                          .text = get_optional_field<string>(json, "text", caption),
                          .received_date = get_required_field<int64_t>(json, "date"),
                          .attachments = parse_attachments(json),
                          .forwarded_from = parse_forwarded_from(json) });
    }

    vector<attachment_meta> parse_attachments(json_value json)
    {
        vector<attachment_meta> res;

        for (auto&& type : settings.allowed_attachment_types)
        {
            if (json.has_member(type))
            {
                attachment_meta meta;
                if (type == "photo")
                {
                    meta = parse_photo_meta(json[type]);
                }
                else if (type == "voice")
                {
                    meta = parse_voice_meta(json[type]);
                }
                else if (type == "video_note")
                {
                    meta = parse_video_note_meta(json[type]);
                }
                else if (type == "sticker")
                {
                    meta = parse_sticker_meta(json[type]);
                }
                else
                {
                    meta = parse_attachment_meta(json[type]);
                }
                res.emplace_back(meta);
            }
        }

        return res;
    }

    attachment_meta parse_photo_meta(json_value json)
    {
        auto biggest_photo_json = select_biggest_photo(json);
        return { .id = get_required_field<string>(biggest_photo_json, "file_id"),
                 .file_name = "photo.jpg",
                 .mime_type = "image/jpeg",
                 .size = get_optional_field<uint64_t>(biggest_photo_json, "file_size") };
    }

    json_value select_biggest_photo(json_value json)
    {
        size_t biggest_width = 0;
        json_value biggest_photo;
        for (auto it = json.array_begin(); it != json.array_end(); ++it)
        {
            auto photo_json = *it;
            auto current_width = get_required_field<size_t>(photo_json, "width");
            if (current_width > biggest_width)
            {
                biggest_photo = photo_json;
                biggest_width = current_width;
            }
        }
        return biggest_photo;
    }

    attachment_meta parse_voice_meta(json_value json)
    {
        return { .id = get_required_field<string>(json, "file_id"),
                 .file_name = "voice.ogg",
                 .mime_type = get_optional_field<string>(json, "mime_type"),
                 .size = get_optional_field<uint64_t>(json, "file_size") };
    }

    attachment_meta parse_video_note_meta(json_value json)
    {
        return { .id = get_required_field<string>(json, "file_id"),
                 .file_name = "video_note",
                 .size = get_optional_field<uint64_t>(json, "file_size") };
    }

    attachment_meta parse_sticker_meta(json_value json)
    {
        return { .id = get_required_field<string>(json, "file_id"),
                 .file_name = "sticker",
                 .size = get_optional_field<uint64_t>(json, "file_size") };
    }

    attachment_meta parse_attachment_meta(json_value json)
    {
        return { .id = get_required_field<string>(json, "file_id"),
                 .file_name = get_optional_field<string>(json, "file_name"),
                 .mime_type = get_optional_field<string>(json, "mime_type"),
                 .size = get_optional_field<uint64_t>(json, "file_size") };
    }

    optional<message_author> parse_forwarded_from(json_value json)
    {
        if (json.has_member("forward_from"))
        {
            return parse_name(json["forward_from"]);
        }
        if (json.has_member("forward_from_chat"))
        {
            return parse_title(json["forward_from_chat"]);
        }
        return {};
    }

    message_author parse_name(json_value json)
    {
        auto name = get_required_field<string>(json, "first_name");
        auto last_name = get_optional_field<string>(json, "last_name");
        name = concat_words(name, last_name);

        auto login = get_optional_field<string>(json, "username");

        return message_author{
            .name = name,
            .login = login,
            .profile_link = make_profile_link(login),
        };
    }

    message_author parse_title(json_value json)
    {
        auto title = get_optional_field<string>(json, "title");
        auto login = get_optional_field<string>(json, "username");

        return message_author{
            .name = title,
            .login = login,
            .profile_link = make_profile_link(login),
        };
    }

    i18n::language parse_language(json_value json)
    {
        auto lang = get_optional_field<string>(json, "language_code", "ru");
        return settings.russian_locale_languages.contains(lang) ? i18n::language::ru :
                                                                  i18n::language::en;
    }

    string concat_words(string prefix, string word)
    {
        if (prefix.empty()) return word;
        if (word.empty()) return prefix;

        return prefix + " " + word;
    }

    string make_profile_link(string login)
    {
        return (boost::format(settings.link_template) % login).str();
    }
};

using module = module_impl<ymod_webserver::server, yhttp::cluster_client>;

}
