#pragma once

#include <common/json.h>
#include <common/types.h>
#include <ymod_webserver/response.h>

namespace botserver::messenger::telegram {

using update_handler = function<future<void>(task_context_ptr, json_value)>;

template <typename Webserver>
class webhook_update_receiver
{
    using webserver_ptr = shared_ptr<Webserver>;

    webserver_ptr webserver;
    update_handler handler;

public:
    webhook_update_receiver(webserver_ptr webserver, string webhook_path) : webserver(webserver)
    {
        if (webserver)
            webserver->bind("", { webhook_path }, [this](ymod_webserver::http::stream_ptr stream) {
                process_request(stream);
            });
    }

    void set_update_handler(update_handler handler)
    {
        this->handler = handler;
    }

private:
    void process_request(ymod_webserver::http::stream_ptr stream)
    {
        auto request = stream->request();
        if (request->content.type != "application" || request->content.subtype != "json")
        {
            LINFO_(stream->ctx()) << "webhook bad request: content is not json";
            return stream->result(ymod_webserver::codes::bad_request, "JSON content required");
        }

        json_value json;
        if (auto error = json.parse({ request->raw_body.begin(), request->raw_body.end() }))
        {
            LINFO_(stream->ctx()) << "webhook bad request: " << *error;
            return stream->result(ymod_webserver::codes::bad_request, "JSON parse error");
        }

        if (!json.has_member("update_id"))
        {
            LINFO_(stream->ctx()) << "webhook bad request: no update_id in json";
            return stream->result(ymod_webserver::codes::bad_request, "Malformed JSON");
        }

        handler(stream->ctx(), json).then([stream](auto future) {
            try
            {
                future.get();
                stream->result(ymod_webserver::codes::ok, "OK");
            }
            catch (const json_field_exception& e)
            {
                LERR_(stream->ctx()) << "exception occured: " << e.what();
                stream->result(ymod_webserver::codes::bad_request, "Malformed JSON");
            }
            catch (const exception& e)
            {
                LERR_(stream->ctx()) << "exception occured: " << e.what();
                stream->result(ymod_webserver::codes::internal_server_error, "Exception occured");
            }
        });
    }
};

}
