#pragma once

#include <common/types.h>
#include <ymod_httpclient/cluster_client.h>
#include <ymod_webserver/server.h>
#include <yplatform/module.h>
#include <yplatform/util/weak_bind.h>
#include <yplatform/util/sstream.h>

namespace botserver::proxy {

struct settings
{
    map<string, string> host_by_prefix;
    set<string> remove_request_headers;

    void parse_ptree(yplatform::ptree conf)
    {
        yplatform::read_ptree(host_by_prefix, conf, "proxy");
        yplatform::read_ptree(remove_request_headers, conf, "remove_request_headers");
    }
};

inline settings make_settings(yplatform::ptree conf)
{
    settings ret;
    ret.parse_ptree(conf);
    return ret;
}

struct module : yplatform::module
{
    using http_client_ptr = shared_ptr<yhttp::cluster_client>;
    map<string, http_client_ptr> http_clients;
    vector<string> all_prefixes;
    settings settings;

    module(yplatform::reactor& reactor, yplatform::ptree conf) : settings(make_settings(conf))
    {
        yhttp::cluster_client::settings http_settings;
        http_settings.parse_ptree(conf.get_child("http"));
        for (auto&& [prefix_path, host] : settings.host_by_prefix)
        {
            auto prefix = "/" + prefix_path;
            all_prefixes.push_back(prefix);
            http_settings.nodes.clear();
            http_settings.nodes.push_back(host);
            http_clients[prefix] = make_shared<yhttp::cluster_client>(reactor, http_settings);
        }
    }

    void init()
    {
        auto webserver = find_module<ymod_webserver::server>("web_server");
        webserver->set_custom_key_extractor("", [all_prefixes = all_prefixes](auto stream) {
            auto default_key = stream->request()->url.make_full_path();
            for (auto&& prefix : all_prefixes)
            {
                if (default_key.starts_with(prefix))
                {
                    return prefix;
                }
            }
            return default_key;
        });
        webserver->bind(
            "",
            all_prefixes,
            yplatform::weak_bind(&module::proxy_request, shared_from(this), ph::_1));
    }

    void proxy_request(ymod_webserver::http::stream_ptr stream)
    {
        forward_request(
            stream->request(), [stream, capture_self](error_code ec, yhttp::response http_resp) {
                if (ec) return stream->result(ymod_webserver::codes::internal_server_error, "");

                self->forward_response(stream, http_resp);
            });
    }

    template <typename Handler>
    void forward_request(ymod_webserver::request_ptr req, Handler h)
    {
        auto uri = yplatform::util::split(req->raw_request_line, " ").at(1);
        auto current_prefix = get_request_prefix(uri);
        if (current_prefix.empty())
        {
            return h(boost::asio::error::not_found, yhttp::response{});
        }

        uri = "/" + uri.erase(0, current_prefix.size());
        std::string headers;
        yplatform::sstream headers_stream(headers);
        for (auto& [name, value] : req->headers)
        {
            if (!settings.remove_request_headers.count(name))
            {
                headers_stream << name << ": " << value << "\r\n";
            }
        }
        auto fwd_request = req->method == ymod_webserver::methods::mth_post ?
            yhttp::request::POST(uri, headers, { req->raw_body.begin(), req->raw_body.end() }) :
            yhttp::request::GET(uri, headers);

        http_clients[current_prefix]->async_run(req->ctx(), fwd_request, h);
    }

    void forward_response(ymod_webserver::http::stream_ptr stream, yhttp::response http_resp)
    {
        stream->set_code(ymod_webserver::codes::code(http_resp.status));
        stream->result_body(http_resp.body);
    }

    string get_request_prefix(string uri)
    {
        string res = "";
        for (auto&& prefix : all_prefixes)
        {
            if (uri.starts_with(prefix))
            {
                res = prefix;
            }
        }
        return res;
    }
};

}
