#pragma once

#include <string>
#include <iostream>
#include <vector>
#include <cctype>

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wconversion"
#pragma GCC diagnostic ignored "-Wold-style-cast"

namespace pystrutils
{
    class string;
    /// validate std::string
    std::string get_validated_utf8( const std::string & );
}

pystrutils::string& operator*= (pystrutils::string &a, size_t N);
pystrutils::string operator* (const pystrutils::string &a, size_t N);
pystrutils::string operator* (size_t N, const pystrutils::string &a);


namespace pystrutils
{
    class string: public std::string
    {
    public:
        string (): std::string() {}
        string (const std::string &a): std::string (a) {}
        string (const char *a): std::string (a) {}
        string (const char *a, size_t length): std::string (a, length) {}

        string capitalize () const
        {
            if (!this->empty())
            {
                string tmp = *this;
                tmp[0] = toupper (tmp[0]);
                return tmp;
            }
            return "";
        }

        string center (size_t n, char fill = ' ') const
        {
            if (n <= this->length())
                return *this;
            else
            {
                string tmp = *this;
                size_t pos, i;
                pos = (n - tmp.length() + 1)/2;
                tmp.resize(pos);
                for (i = 0; i < pos; i++)
                    tmp[i] = fill;
                tmp.append(*this);
                i = tmp.length();
                tmp.resize(n);
                while (i < n)
                    tmp[i++] = fill;
                return tmp;
            }
        }

        string slice (int start) const
        {
            if (start < 0)
            {
                if (-start >= int(this->length()))
                    return *this;
                return this->substr(this->length() + start, -start);
            }
            if (start >= int(this->length()))
                return "";
            return this->substr(start, this->length() - start);
        }

        string slice (int start, int end) const
        {
            if (start < 0)
            {
                start = this->length() + start;
                if (start < 0)
                    start = 0;
            }
            if (end < 0)
            {
                end = this->length() + end;
                if (end < 0)
                    end = this->length();
            }
            if (end <= start)
                return "";
            if (start >= (int)this->length())
                return "";
            return this->substr(start, end - start);
        }

        size_t count (string sub) const
        {
            size_t pos     = 0;
            size_t counter = 0;
            while (pos != string::npos)
            {
                pos = this->find (sub, pos);
                if (pos != string::npos)
                {
                    counter ++;
                    pos += sub.length();
                }
            }
            return counter;
        }

        bool endswith (string sub) const
        {
            if (this->length() < sub.length())
                return false;
            return this->slice(-sub.length()) == sub;
        }

        bool startswith (string sub) const
        {
            if (this->length() < sub.length())
                return false;
            return this->slice(0, sub.length()) == sub;
        }

        string expandtabs (size_t tabwidth = 8) const
        {
            string tmp         = "";
            string replacement = string (" ") * tabwidth;
            for (size_t i = 0; i < this->length(); i++)
                if (this->at(i) != '\t')
                    tmp.append(this->slice(i, i+1));
                else
                    tmp.append(replacement);
            return tmp;
        }

        bool isalnum () const
        {
            for (size_t i = 0; i < this->length(); i++)
                if (! (std::isalpha(this->at(i)) || std::isdigit( this->at(i))))
                    return false;
            return true;
        }

        bool isalpha () const
        {
            for (size_t i = 0; i < this->length(); i++)
                if (! (std::isalpha(this->at(i))))
                    return false;
            return true;
        }

        bool isdigit () const
        {
            if (this->empty())
                return false;
            for (size_t i = 0; i < this->length(); i++)
                if (! (std::isdigit(this->at(i))))
                    return false;
            return true;
        }

        bool istitle () const
        {
            if (this->empty())
                return false;
            return *this == this->title ();
        }

        bool isupper () const
        {
            for (size_t i = 0; i < this->length(); i++)
                if (std::isalpha(this->at(i)) && !std::isupper( this->at(i)))
                    return false;
            return true;
        }

        bool islower () const
        {
            for (size_t i = 0; i < this->length(); i++)
                if (std::isalpha(this->at(i)) && !std::islower( this->at(i)))
                    return false;
            return true;
        }

        template <class T> string join (T start, T finish) const
        {
            bool first = true;
            string tmp;
            for (T it = start; it != finish; it++)
            {
                if (!first)
                {
                    tmp.append(*this);
                    tmp.append(*it);
                }
                else
                {
                    tmp.append(*it);
                    first = false;
                }
            }
            return tmp;
        }

        string ljust (size_t width, char fill = ' ') const
        {
            if (width <= this->length())
                return *this;
            size_t pos = this->length();
            string tmp = *this;
            tmp.resize(width);
            for (; pos < tmp.length(); pos++)
                tmp[pos] = fill;
            return tmp;
        }

        string rjust (size_t width, char fill = ' ') const
        {
            if (width <= this->length())
                return *this;
            size_t len = width - this->length();
            string tmp;
            tmp.resize(len);
            for (size_t i = 0; i < len; i++)
                tmp[i] = fill;
            tmp += *this;
            return tmp;
        }

        string lower () const
        {
            string tmp = *this;
            for (size_t i = 0; i < tmp.length(); i++)
            {
                if (std::isalpha(tmp[i]))
                    tmp[i] = tolower (tmp[i]);
            }
            return tmp;
        }

        string upper () const
        {
            string tmp = *this;
            for (size_t i = 0; i < tmp.length(); i++)
            {
                if (std::isalpha(tmp[i]))
                    tmp[i] = toupper (tmp[i]);
            }
            return tmp;
        }

        string lstrip (const char* pattern = " \r\n\t") const
        {
            size_t pos = 0;
            while (pos < this->length())
            {
                size_t i = 0;
                char sym = this->at(pos);
                while (pattern[i] != sym && pattern[i] != 0)
                    i++;
                if (pattern[i] == 0)
                {
                    if (pos == 0)
                        return *this;
                    else
                        return this->substr(pos, this->length() - pos);
                }
                pos++;
            }
            return "";
        }

        string rstrip (const char* pattern = " \r\n\t") const
        {
            size_t pos = this->length();
            while (pos > 0)
            {
                pos--;
                size_t i = 0;
                char sym = this->at(pos);
                while (pattern[i] != sym && pattern[i] != 0)
                    i++;
                if (pattern[i] == 0)
                {
                    if (pos == this->length() - 1)
                        return *this;
                    else
                        return this->substr(0, pos + 1);
                }
            }
            return "";
        }

        string strip (const char *pattern = " \r\n\t") const
        {
            return this->lstrip(pattern).rstrip(pattern);
        }

        std::vector<string> partition (const char separator) const
        {
            std::vector<string> result;
            result.resize(3, "");
            size_t pos = this->find(separator, 0);
            result[0] = slice (0, pos);
            result[1] = slice (pos, pos + 1);
            result[2] = slice (pos + 1);
            return result;
        }

        std::vector<string> rpartition (const char separator) const
        {
            std::vector<string> result;
            result.resize(3, "");
            size_t pos = this->rfind(separator, this->length() - 1);
            result[0] = slice (0, pos);
            result[1] = slice (pos, pos + 1);
            result[2] = slice (pos + 1);
            return result;
        }

        string replace (string old, string newstring) const
        {
            size_t pos = 0;
            string tmp = *this;
            while (pos != string::npos)
            {
                pos = tmp.find(old, pos);
                if (pos != string::npos)
                {
                    tmp.erase(pos, old.length());
                    tmp.insert(pos, newstring);
                    pos += newstring.length();
                }
            }
            return tmp;
        }

        string title () const
        {
            string tmp    = *this;
            bool newstate = true;
            for (size_t i = 0; i < tmp.length(); i++)
            {
                if (std::isspace (tmp[i]))
                    newstate = true;
                if (std::isalpha (tmp[i]) && newstate)
                    tmp[i] = std::toupper (tmp[i]);
                newstate = false;
            }
            return tmp;
        }

        std::vector<string> split (const char *separators = " \t\r\n") const
        {
            std::vector<string> res;
            if (this->empty())
            {
                res.push_back("");
                return res;
            }
            for (size_t ppos = 0, pos; ppos < this->length(); ppos = pos + 1)
            {
                char sep;
                bool break_loop = false;
                for (pos = ppos; pos < this->length(); pos++)
                {
                    size_t i = 0;
                    char sym = this->at(pos);
                    for (i = 0; sep = separators[i], sep != 0; i++)
                        if (sym == sep)
                        {
                            break_loop = true;
                            break;
                        }
                    if (break_loop)
                        break;
                }
                if (sep == 0)
                {
                    pos = this->length();
                }
                res.push_back(this->substr(ppos, pos - ppos));
            }
            return res;
        }

        string swapcase () const
        {
            string tmp = *this;
            for (size_t i = 0; i < tmp.length(); i++)
            {
                if (std::isalpha(tmp[i]))
                {
                    tmp[i] = (std::isupper(tmp[i])?(std::tolower(tmp[i])):(std::toupper(tmp[i])));
                }
            }
            return tmp;
        }
    };
}

std::ostream& operator<< (std::ostream& bugaga, std::vector<pystrutils::string> result);

#pragma GCC diagnostic pop
