#pragma once

#include <string>
#include <type_traits>
#include <boost/hana.hpp>

namespace crypto {

using namespace boost::hana::literals;

template<typename Name, typename Size>
struct Key {
    constexpr static Name name;
    constexpr static Size size;

    Key() = default;

    explicit Key(std::string key)
            : _value(std::move(key))
    {
        using namespace std::literals;
        if (_value.size() != size) {
            throw std::invalid_argument("Wrong size of "s + boost::hana::to<const char*>(name) +  "key: expect " + std::to_string(size)
                                        + " byte, but actual " + std::to_string(_value.size()) + " byte");
        }
    }

    operator const std::string&() const {
        return _value;
    }

    const unsigned char * data() const {
        return reinterpret_cast<const unsigned char *>(_value.data());
    }

private:
    std::string _value;
};

using HmacKey = Key<decltype(boost::hana::string_c<'h', 'm', 'a', 'c'>), std::integral_constant<std::size_t, 256/8>>;
using AesKey = Key<decltype(boost::hana::string_c<'a','e','s'>), std::integral_constant<std::size_t, 256/8>>;

std::string hmac(const std::string& str, const std::string& key);

std::string encrypt_string(const std::string& in, const std::string& key, const std::string& iv);

std::string decrypt_string(const std::string& in, const std::string& key, const std::string& iv);


using blob = std::vector<std::uint8_t>;

blob generateIv(std::size_t length);

blob aesEncrypt(std::string_view in, const AesKey& key, const blob& iv);

std::string aesDecrypt(const blob &in, const AesKey& key, const blob& iv);

}
