#pragma once

#include <boost/algorithm/string.hpp>
#include <butil/butil.h>
#include <butil/key_value.h>
#include <butil/split_key_value.h>
#include <butil/http/url.h>
#include <sstream>

struct  HttpArguments {
    using Argument = ParameterList;
    using ArgumentVector = MultiKeyValue;
    using FlatArguments = std::map<std::string, std::string>;

    class InvalidArgument: public std::logic_error {
    public:
        InvalidArgument(const std::string& msg)
            : std::logic_error(msg)
        {}
    };

    void add(const std::string& name, const std::string& value) {
        if (!name.empty()) {
            this->arguments[name].push_back(value);
        } else {
            throw InvalidArgument("trying to add argument with empty name");
        }
    }

    void fromUrl( const std::string & url) {
        std::vector<std::string> params;
        if (const auto pos = url.find('?'); pos != std::string::npos) {
            std::string_view query = std::string_view(url).substr(pos + 1);
            boost::split(params, query, boost::is_any_of("&"));
        }
        for (auto& param : params) {
            std::pair<std::string,std::string> kv = splitKeyValue(param);
            if (!kv.first.empty()) {
                arguments[decode_url(kv.first)].push_back(decode_url(kv.second));
            }
        }
    }
    std::string format() const {
        typedef ArgumentVector::const_iterator Iterator;
        bool firstarg=true;
        std::ostringstream result;
        for (Iterator it=arguments.begin(); it!=arguments.end(); ++it) {
            typedef Argument::const_iterator AIterator;
            const Argument& arg=it->second;
            for (AIterator jt=arg.begin(); jt!=arg.end(); ++jt) {
                if (!firstarg) {
                    result << "&";
                } else {
                    firstarg=false;
                }
                result << encode_url(it->first) << "=" << encode_url(*jt);
            }
        }
        return result.str();
    }

    FlatArguments flatten() const {
        FlatArguments res;

        for (const auto& args: arguments) {
            if (args.second.size() > 1) {
                std::ostringstream out;
                out << "cannot make arguments flat: key=" << args.first << " size=" << args.second.size();
                throw std::runtime_error(out.str());
            } else {
                res[args.first] = args.second.empty() ? "" : args.second.front();
            }
        }

        return res;
    }

    ArgumentVector arguments;
};

namespace http {

inline std::string combineArgs(const url& u, const HttpArguments& args, bool prefixQuestion = true) {
    const size_t argsStart = u.uri().find('?');
    const std::string urlArgs = (argsStart == std::string::npos) ? "" : u.uri().substr(argsStart+1);
    const std::string argsStr = args.format();
    std::stringstream ss;
    if (prefixQuestion && (urlArgs.size() || argsStr.size())) {
        ss << '?';
    }
    ss << urlArgs;
    if (urlArgs.size() && argsStr.size()) {
        ss << '&';
    }
    ss << argsStr;
    return ss.str();
}

}
