#ifndef RFC2822_H_2013_11_26
#define RFC2822_H_2013_11_26

#include <string>
#include <list>
#include <stdexcept>

#include <iostream>
#include <memory>
#include <optional>

#include <recipient_parser/types/mailbox_group.hpp>

namespace rfc2822ns {

    using std::string;
    using std::list;
    typedef string::size_type size_type;

    struct invalid_address: public std::runtime_error
    {
        invalid_address(const std::string& what): std::runtime_error(what) {}
    };

    string quote(const string&);
    string quote(const list<string>&);
    string paste(const list<string>&);
    string join_address(const string&, const string&);
    string join_address(const string&, const string&, const string&);
    bool unquote_pair(const string&, size_type&, string&);
    bool quoted_pair(const string&, size_type&, string&);
    bool fws(const string&, size_type&, string&);
    bool comment(const string&, size_type&, string&);
    bool quoted_string(const string&, size_type&, string&);
    bool atom(const string&, size_type&, string&);
    bool dotatom(const string&, size_type&, string&);
    bool xatom(const string&, size_type&, string&);
    bool display_name(const string&, size_type&, list<string>&);
    bool domain_literal(const string&, size_type&, string&);
    bool addrspec(const string&, size_type&, string&, string&);
    bool addrspec(const string&, size_type&, string&);
    bool angleaddr(const string&, size_type&, string&, string&);
    bool angleaddr(const string& src, size_type pos, string& dst);
    bool address(const string&, size_type&, list<string>&, string&, string&);
    bool address(const string&, size_type&, string&, string&);

    class old_address_iterator
    {
        static const string empty;
        const string& src_;
        size_type pos_;
        string display_, local_, domain_, pretty_, address_;

    public:
        old_address_iterator(void): src_(old_address_iterator::empty), pos_(string::npos)
        {
        }

        old_address_iterator(const std::string& src): src_(src), pos_(0)
        {
            operator++();
        }

        bool operator == (const old_address_iterator& x)
        {
            return (&src_ == &x.src_ && pos_ == x.pos_)
                || ((&src_ == &old_address_iterator::empty
                     || &x.src_ == &old_address_iterator::empty)
                    && pos_ == string::npos && x.pos_ == string::npos);
        }

        bool operator != (const old_address_iterator& x)
        {
            return !operator==(x);
        }

        old_address_iterator& operator++(void);

        old_address_iterator operator++(int)
        {
            old_address_iterator tmp(*this);
            operator++();
             return tmp;
        }

        const string& display(void) const { return display_; }
        const string& pretty(void) const {return pretty_; }
        const string& local(void) const { return local_; }
        const string& domain(void) const { return domain_; }
        string address(void) const { return address_; }
    };

    class new_address_iterator {
        using Recipients = rcpt_parser::types::MailboxGroup::GroupList;
        Recipients recipients;
        Recipients::const_iterator current;

        new_address_iterator(Recipients&& recipients);

    public:
        new_address_iterator();

        static std::optional<new_address_iterator> parse(const std::string& src);

        bool operator == (const new_address_iterator& other);
        bool operator != (const new_address_iterator& other);

        new_address_iterator& operator++(void);
        new_address_iterator operator++(int)
        {
            new_address_iterator tmp(*this);
            operator++();
             return tmp;
        }

        string display() const;
        string pretty() const;
        const string& local() const;
        const string& domain() const;
        string address() const;
    };

    class address_iterator {
        struct address_iterator_base {
            virtual ~address_iterator_base() {}

            virtual string display() const = 0;
            virtual string pretty() const = 0;
            virtual string local() const = 0;
            virtual string domain() const = 0;
            virtual string address() const = 0;

            virtual void advance() = 0;
            virtual bool empty() = 0;
        };

        template<typename IteratorImpl>
        struct address_iterator_impl : address_iterator_base {
            IteratorImpl impl;

            template<typename ... Args>
            address_iterator_impl(Args&& ... args)
                    : impl(std::forward<Args>(args)...) {}

            string display() const override {
                return impl.display();
            }
            string pretty() const override {
                return impl.pretty();
            }
            string local() const override {
                return impl.local();
            }
            string domain() const override {
                return impl.domain();
            }
            string address() const override {
                return impl.address();
            }

            void advance() override {
                ++impl;
            }
            bool empty() override {
                return impl == IteratorImpl();
            }
        };

        bool empty() const {
            return !impl || impl->empty();
        }

        std::shared_ptr<address_iterator_base> impl;
    public:
        address_iterator();
        address_iterator(const address_iterator& other) {
            impl = other.impl;
        }
        address_iterator(const std::string& src);

        bool operator == (const address_iterator& other);
        bool operator != (const address_iterator& other) {
            return !(*this == other);
        }

        address_iterator& operator++(void);
        address_iterator operator++(int)
        {
            address_iterator tmp(*this);
            operator++();
            return tmp;
        }

        string display(void) const;
        string pretty(void) const;
        string local(void) const;
        string domain(void) const;
        string address(void) const;
    };
}


#endif // RFC2822_H_2013_11_26
