#ifndef _EXPAT_TRANSFORMER_H_2013_11_23
#define _EXPAT_TRANSFORMER_H_2013_11_23

#include <vector>
#include <string>
#include <memory>
#include <boost/utility.hpp>
#include <boost/lexical_cast.hpp>
#include <expat.h>

template <class Iterator, class Handler>
class ExpatTransformer : private boost::noncopyable {
    public:
        ExpatTransformer(const Iterator& begin,
                         const Iterator& end,
                         Handler& handler);
        ~ExpatTransformer();
        Handler& handler() {
            return m_handler;
        }
        XML_Parser& parser() {
            return m_xmlParser;
        }
        const Iterator& begin() {
            return m_begin;
        }
        const Iterator& end() {
            return m_end;
        }
        bool transform();
        std::string error();
        static ExpatTransformer& unWrap(XML_Parser& parser);
    private:
        Iterator m_begin;
        Iterator m_end;
        Handler& m_handler;
        XML_Parser m_xmlParser;
};

template <class Iterator, class Handler>
ExpatTransformer<Iterator, Handler>::
ExpatTransformer(const Iterator& begin, const Iterator& end, Handler& handler)
    : m_begin(begin)
    , m_end(end)
    , m_handler(handler)
{
    m_xmlParser=XML_ParserCreate("UTF-8");
    XML_UseForeignDTD(m_xmlParser, true);
    XML_SetUserData(m_xmlParser, this);
}

template <class Iterator, class Handler>
ExpatTransformer<Iterator, Handler>::
~ExpatTransformer() {
    XML_ParserFree(m_xmlParser);
}

template <class Iterator, class Handler>
bool
ExpatTransformer<Iterator, Handler>::
transform() {
    XML_SetStartElementHandler(m_xmlParser, Handler::startElement);
    XML_SetEndElementHandler(m_xmlParser, Handler::endElement);
    XML_SetDefaultHandler(m_xmlParser, Handler::defaultHandler);
    XML_Status status=
            XML_Parse(m_xmlParser, &(*m_begin), static_cast<int>(m_end-m_begin), 1/*is final*/);
    return (XML_STATUS_OK==status);
}

template <class Iterator, class Handler>
std::string
ExpatTransformer<Iterator, Handler>::
error() {
    XML_Size line=XML_GetCurrentLineNumber(m_xmlParser);
    XML_Size column=XML_GetCurrentColumnNumber(m_xmlParser);
    return
        std::string(XML_ErrorString(XML_GetErrorCode(m_xmlParser)))
      + " [line: " + boost::lexical_cast<std::string>(line)+", "
      + " column: " + boost::lexical_cast<std::string>(column)+"]";
}

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wold-style-cast"

template <class Iterator, class Handler>
ExpatTransformer<Iterator, Handler>&
ExpatTransformer<Iterator, Handler>::
unWrap(XML_Parser& parser) {
    return *(static_cast<ExpatTransformer>(XML_GetUserData(parser)));
}

#pragma GCC diagnostic pop

#endif
