#include <butil/StrUtils/Iconv.h>
#include <errno.h>

/*

$Log$
Revision 1.4  2007/07/04 11:07:48  virtan
iconv buffer increased, extra assert added

Revision 1.3  2005/11/24 14:12:28  virtan
no warnings anymore

Revision 1.2  2005/08/12 09:08:03  virtan
folder name recoding when filling folder_set

Revision 1.1.1.1  2005/03/10 11:06:40  virtan
Initial importing


*/

static const size_t bufferSize = 16384;

Iconv::Iconv(const std::string &from, const std::string &to, bool recover) :
    ip(iconv_t(-1)),memBuf(),modeRecover(recover)
{
    direction(from,to);
}

Iconv::~Iconv() {
    if(!iserror())
        iconv_close(ip);
}

void Iconv::direction(const std::string &from, const std::string &to) {
    if(!iserror())
        iconv_close(ip);
    ip=iconv_open(to.c_str(),from.c_str());
    if(!iserror()) return;
    std::string tunedFrom(from),tunedTo(to);
    tuneCharset(tunedFrom);
    tuneCharset(tunedTo);
    ip=iconv_open(tunedTo.c_str(),tunedFrom.c_str());
}

void Iconv::tuneCharset(std::string &charset) {
    if (charset.find("1251")!=std::string::size_type(-1)) {
        charset="cp1251";
        return;
    }
    if ("ks_c_5601-1987"==charset) {
        charset="EUCKR";
        return;
    }
}

void Iconv::recover(bool value) {
    modeRecover=value;
}

bool Iconv::recover() {
    return modeRecover;
}

int Iconv::recode(const std::string &source, std::string &dest) {
    if(iserror())
        return ICONV_BADDIRECTION;
    char *inbuf=const_cast<char*>(source.c_str());
    if (!memBuf) {
        memBuf = std::make_unique<char[]>(bufferSize);
    }
    char* const buf = memBuf.get();
    char* outbuf = memBuf.get();
    size_t inbl=source.size();
    size_t oubl = bufferSize;

    while(true) {
        const int res = int(iconv(ip,&inbuf,&inbl,&outbuf,&oubl));
        if(oubl>bufferSize)
            oubl=0;
        if(res>=0) {
            dest.append(buf,bufferSize-oubl);
            if(inbl!=0)
                return ICONV_UNKNOWN;
            return ICONV_OK;
        } else {
            if(errno==EILSEQ) {
                if(modeRecover) {
                    inbuf++;
                    inbl--;
                    continue;
                } else {
                    dest.append(buf,bufferSize-oubl);
                    return int(source.size()-inbl + 1); // ICONV_BADSEQUENCE
                }
            } else if(errno==EINVAL) {
                dest.append(buf,bufferSize-oubl);
                return int(source.size()-inbl + 1); // ICONV_BADSEQUENCE or ICONV_NEEDMORE
            } else if(errno==E2BIG) {
                dest.append(buf,bufferSize-oubl);
                outbuf=buf;
                oubl=bufferSize;
                continue;
            } else {
                dest.append(buf,bufferSize-oubl);
                return ICONV_UNKNOWN;
            }
        }
    }
}

std::string Iconv::recode(const std::string &source) {
    std::string result;
    this->recode(source,result);
    return result;
}

std::string Iconv::recode(const std::string &src, char repl) {
    std::string dst;
    int i = 0, j = recode(src, dst);
    while(j != ICONV_OK) {
        dst+= repl; i += j;
        j = recode(src.substr(i, src.size() - i), dst);
    }
    return dst;
}

bool Iconv::iserror() {
    return ip==iconv_t(-1);
}

const std::string Iconv::error(int errorCode) {
    std::string errorStr;
    switch(errorCode) {
        case ICONV_OK:
            errorStr.assign("no error");
            break;
        case ICONV_BADDIRECTION:
            errorStr.assign("unknown direction requested");
            break;
        case ICONV_UNKNOWN:
            errorStr.assign("unknown error");
            break;
        default:
            errorStr.assign("bad sequence met");
            break;
    }
    return errorStr;
}

int Iconv::recode(const std::string &from, const std::string &to,
        const std::string &source, std::string &dest, bool recover)
{
    Iconv ic(from,to,recover);
    return ic.recode(source,dest);
}
