#include <cstring>
#include <vector>
#include <stdexcept>
#include <sstream>

#include <openssl/evp.h>
#include <openssl/hmac.h>
#include <openssl/err.h>
#include <openssl/rand.h>

#include <butil/crypt.h>

namespace {

struct Closure {
    void* data;
    int (*function)(const char *, size_t, void *);
};

template <class Function>
Closure makeClosure(Function& function) {
    struct Wrap {
        static int call(const char *str, size_t size, void *data) {
            return reinterpret_cast<Function *>(data)->operator()(str, size);
        }
    };
    return Closure {&function, &Wrap::call};
}

using EVP_CIPHER_CTX_unique_ptr = std::unique_ptr<EVP_CIPHER_CTX, decltype(&::EVP_CIPHER_CTX_free)>;

} // namespace

namespace crypto {

#define BUTIL_HANDLE_OPENSSL_ERROR(expression) \
    do { \
        if (!(expression)) { \
            std::ostringstream stream; \
            stream << "OpenSSL error when call " << #expression << ": "; \
            auto function = [&] (auto str, auto size) { stream.write(str, size); return 1; }; \
            const auto closure = makeClosure(function); \
            ERR_print_errors_cb(closure.function, closure.data); \
            stream << " at " << __FILE__ << ":" << __LINE__; \
            throw std::runtime_error(stream.str()); \
        } \
    } while (false)

std::string hmac(const std::string& str, const std::string& key) {
    std::vector<unsigned char> out(EVP_MAX_MD_SIZE);
    unsigned int len;
    BUTIL_HANDLE_OPENSSL_ERROR(HMAC(
        EVP_sha256(),
        reinterpret_cast<const void*>(key.c_str()),
        int(key.size()),
        reinterpret_cast<const unsigned char*>(str.c_str()),
        int(str.size()),
        out.data(),
        &len
    ));
    return std::string(reinterpret_cast<const char*>(out.data()), std::size_t(len));
}

blob generateIv(std::size_t length) {
    blob iv(length, 0);

    BUTIL_HANDLE_OPENSSL_ERROR(
            1 == RAND_bytes(iv.data(), static_cast<int>(iv.size()))
    );

    return iv;
}

class BlowfishCipher {
public:
    BlowfishCipher() {
        ctx_ = EVP_CIPHER_CTX_new();
    }

    ~BlowfishCipher() {
        EVP_CIPHER_CTX_free(ctx_);
    }

    std::string encrypt(const std::string& in, const std::string& key, const std::string& iv) {
        BUTIL_HANDLE_OPENSSL_ERROR(EVP_EncryptInit_ex(ctx_, EVP_bf_cbc(), NULL,
           reinterpret_cast<const unsigned char*>(key.c_str()),
           reinterpret_cast<const unsigned char*>(iv.c_str())));
        std::vector<unsigned char> out(in.size() + EVP_CIPHER_block_size(EVP_bf_cbc()) + 1024);
        int outlen = 0;
        int tmplen = 0;
        BUTIL_HANDLE_OPENSSL_ERROR(EVP_EncryptUpdate(ctx_, &out[0], &outlen,
            reinterpret_cast<const unsigned char *>(in.data()), int(in.size())));
        BUTIL_HANDLE_OPENSSL_ERROR(EVP_EncryptFinal_ex(ctx_, &out[outlen], &tmplen));
        outlen += tmplen;
        return std::string(&out[0], &out[outlen]);
    }

    std::string decrypt(const std::string& in, const std::string& key, const std::string& iv) {
        BUTIL_HANDLE_OPENSSL_ERROR(EVP_DecryptInit_ex(ctx_, EVP_bf_cbc(), NULL,
            reinterpret_cast<const unsigned char*>(key.c_str()),
            reinterpret_cast<const unsigned char*>(iv.c_str())));
        std::vector<unsigned char> out(in.size() + 1024);
        int outlen = 0;
        int tmplen = 0;
        BUTIL_HANDLE_OPENSSL_ERROR(EVP_DecryptUpdate(ctx_, &out[0], &outlen,
                reinterpret_cast<const unsigned char *>(in.data()), int(in.size())));
        BUTIL_HANDLE_OPENSSL_ERROR(EVP_DecryptFinal_ex(ctx_, &out[outlen], &tmplen));
        outlen += tmplen;
        return std::string(&out[0], &out[outlen]);
    }

private:
    EVP_CIPHER_CTX *ctx_;
};

std::string encrypt_string(const std::string& in, const std::string& key, const std::string& iv) {
    return BlowfishCipher().encrypt(in, key, iv);
}

std::string decrypt_string(const std::string& in, const std::string& key, const std::string& iv) {
    return BlowfishCipher().decrypt(in, key, iv);
}

blob aesEncrypt(std::string_view in, const AesKey& key, const blob& iv) {
    EVP_CIPHER_CTX_unique_ptr ctx_(EVP_CIPHER_CTX_new(), ::EVP_CIPHER_CTX_free);

    BUTIL_HANDLE_OPENSSL_ERROR(
        EVP_EncryptInit_ex(ctx_.get(), EVP_aes_256_cbc(), NULL, key.data(), iv.data())
    );

    blob out(in.size() + EVP_CIPHER_block_size(EVP_aes_256_cbc()));
    int outlen = 0;
    BUTIL_HANDLE_OPENSSL_ERROR(EVP_EncryptUpdate(ctx_.get(), &out[0], &outlen,
                               reinterpret_cast<const unsigned char *>(in.data()), static_cast<int>(in.size())));
    int paddinglen = 0;
    BUTIL_HANDLE_OPENSSL_ERROR(EVP_EncryptFinal_ex(ctx_.get(), &out[outlen], &paddinglen));
    outlen += paddinglen;
    out.resize(outlen);
    return out;
}

std::string aesDecrypt(const blob &in, const AesKey& key, const blob& iv) {
    EVP_CIPHER_CTX_unique_ptr ctx_(EVP_CIPHER_CTX_new(), ::EVP_CIPHER_CTX_free);

    BUTIL_HANDLE_OPENSSL_ERROR(
        EVP_DecryptInit_ex(ctx_.get(), EVP_aes_256_cbc(), NULL, key.data(), iv.data())
    );

    std::string out;
    out.resize(in.size() + EVP_CIPHER_block_size(EVP_aes_256_cbc()));
    int outlen = 0;
    BUTIL_HANDLE_OPENSSL_ERROR(EVP_DecryptUpdate(ctx_.get(),
            reinterpret_cast<unsigned char *>(out.data()), &outlen, in.data(), static_cast<int>(in.size()))
    );
    int paddinglen = 0;
    BUTIL_HANDLE_OPENSSL_ERROR(
            EVP_DecryptFinal_ex(ctx_.get(), reinterpret_cast<unsigned char *>(out.data() + outlen), &paddinglen)
    );
    outlen += paddinglen;
    out.resize(static_cast<std::size_t>(outlen));
    return out;
}

#undef BUTIL_HANDLE_OPENSSL_ERROR

} // namespace crypto
