#include <boost/date_time/local_time/local_time.hpp>
#include <butil/datetime/date_utils.h>
#include <library/cpp/geobase/lookup.hpp>
#include <library/cpp/geobase/timezone_getter.hpp>


std::string time2stringFormat ( const time_t tt, const std::string & format )
{
    struct tm brokenDownTime( DateUtils::tmDefault() );
    if(!localtime_r(&tt, &brokenDownTime)) {
        return "";
    }
    char xtc_buf[50];
    size_t sz=strftime(xtc_buf, 50, format.c_str(), &brokenDownTime);
    if (0!=sz) {
        return std::string(xtc_buf, sz);
    }
    return "";
}

std::string time2stringFormatGmt ( const time_t tt, const std::string & format )
{
    struct tm brokenDownTime( DateUtils::tmDefault() );
    if(!gmtime_r(&tt, &brokenDownTime)) {
        return "";
    }
    char xtc_buf[50];
    size_t sz=strftime(xtc_buf, 50, format.c_str(), &brokenDownTime);
    if (0!=sz) {
        return std::string(xtc_buf, sz);
    }
    return "";
}

time_t DateUtils::string2timeFormat ( const std::string & time, const std::string & format )
{
    struct tm ts( DateUtils::tmDefault() );
    time_t result = 0;
    if (strptime(time.c_str(), format.c_str(), &ts) == 0) {
        return result;
    }
    result = mktime(&ts);
    return result;
}

namespace {
time_t get_offset(const std::string& tz, time_t tt)
{
    try {
        #ifdef ARCADIA_BUILD
        NGeobase::NImpl::TTimezoneGetter tzGetter;
        return tzGetter.Count(tz, tt).Offset;
        #else
        return geobase5::timezone_getter::count(tz.c_str(), tt).offset;
        #endif
    } catch(...) {
        try {
            #ifdef ARCADIA_BUILD
            NGeobase::NImpl::TTimezoneGetter tzGetter;
            return tzGetter.Count("Europe/Moscow", tt).Offset;
            #else
            return geobase5::timezone_getter::count("Europe/Moscow", tt).offset;
            #endif
        } catch(...) {
            return 10800;
        }
    }
}
} // unnamed namespace

time_t DateUtils::tztime(time_t tt, const std::string& tz)
{
    tt += get_offset(tz, tt);
    return tt;
}

time_t DateUtils::tztimeUtc(time_t tt, const std::string& tz)
{
    tt -= get_offset(tz, tt);
    return tt;
}

std::string DateUtils::tztime2string(time_t tt, const std::string& tz, const std::string& format) {
    using namespace boost::local_time;

    time_t off = get_offset(tz, tt);
    boost::posix_time::time_duration d(0, 0, boost::posix_time::time_duration::sec_type(off));
    dst_adjustment_offsets adj(boost::posix_time::time_duration(0, 0, 0),
                               boost::posix_time::time_duration(0, 0, 0),
                               boost::posix_time::time_duration(0, 0, 0));
    boost::shared_ptr<dst_calc_rule> no_rules;
    time_zone_names n("", "", "", "");
    time_zone_ptr zone(new custom_time_zone(n, d, adj, no_rules));

    local_date_time t(boost::posix_time::from_time_t(tt), zone);
    local_time_facet *output_facet = new local_time_facet(format.c_str());
    std::ostringstream s;
    s.imbue(std::locale(std::locale::classic(), output_facet));
    s << t;
    return s.str();
}

std::string DateUtils::rfc2822time(time_t tt, const std::string& tz) {
    return tztime2string(tt, tz, "%a, %d %b %Y %H:%M:%S %q");
}
