#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <butil/StrUtils/pystrutils.h>
#include <butil/email/helpers.h>

namespace EmailHelpers {

    namespace {

    Email fromString(const rfc2822ns::address_iterator it, bool lowercase)
    {
        rfc2822ns::address_iterator end;
        if (end != it) {
            std::string domain = idna::encode( it.domain() );
            std::string login = it.local();
            std::string displayName = it.display();

            if (lowercase) {
                utf::to_lower( login );
                utf::to_lower( domain );
            }

            displayName = pystrutils::get_validated_utf8(displayName);

            Email email(login, domain, displayName );
            if (isEmptyAddress(email) || !isValidAddress(email)) {
                throw rfc2822ns::invalid_address("EmailHelpers::fromString: '" +
                    it.display() + "','" + login + "','" + domain + "'");
            }
            return email;
        }
        return Email();
    }

    }

    bool isEmptyAddress(const Email &email)
    {
        return email.local().empty() || email.domain().empty();
    }

    bool isValidAddress(const Email &email)
    {
       if (isEmptyAddress(email)) {
           return false;
       }
       if( is_yandex_host(email.domain()) && !isValidYandexLogin(email.local()) ) {
           return false;
       } else {
           if (email.local().find('%') != std::string::npos) {
             return false;
           }
       }
       return true;
    }

    std::string toString(const Email &email)
    {
        return rfc2822ns::join_address(email.displayName(), email.local(), email.domain());
    }

    inline std::string emailToString(const Email &email)
    {
        return toString(email);
    }

    std::string toString(const EmailVec& vec)
    {
        using boost::adaptors::transformed;
        using boost::join;
        return join( vec | transformed(EmailHelpers::emailToString), ",");
    }

    Email fromString(const rfc2822ns::address_iterator it) {
        return fromString(it, true);
    }

    Email fromStringWithoutLowercase(const rfc2822ns::address_iterator it) {
        return fromString(it, false);
    }

    Email fromString(const std::string &email)
    {
        rfc2822ns::address_iterator it(email);
        return fromString(it);
    }

    Email fromString(const std::string &email, const std::nothrow_t &) {
        try {
            return fromString(email);
        } catch (...) {
        }
        return Email("", "", email);
    }

    EmailVec toEmailVec(const std::string &emails) {
        try {
            EmailVec vEmails;
            for (rfc2822ns::address_iterator it(emails), end; end != it; ++it) {
                vEmails.push_back(fromString(it));
            }
            return vEmails;
        } catch (const std::exception&) {
            EmailVec vEmails;
            vEmails.push_back(Email("", "", emails));
            return vEmails;
        }
    }

    std::string idnaize(const std::string& email) {
        return toString(fromString(email));
    }

    bool isEqualEmails(const Email& email1,const Email& email2)
    {
        if ( (is_yandex_host(email1.domain()) && is_yandex_host(email2.domain())) ||
                (is_yandex_team_host(email1.domain()) && is_yandex_team_host(email2.domain())) ) {
            return loginsEquals(email1.local(), email2.local());
        }
        return boost::algorithm::iequals(email1.local(), email2.local()) &&
               boost::algorithm::iequals(email1.domain(), email2.domain());
    }

    EmailVec::iterator findEqualEmail(EmailVec &vec, const Email& value)
    {
        using std::bind;
        using std::placeholders::_1;
        return find_if(vec.begin(),vec.end(),bind(&isEqualEmails,_1,value));
    }
}
