#include <boost/lexical_cast.hpp>
#include <butil/split_key_value.h>
#include <butil/http/url.h>

namespace http {

url::size_type url::extractLoginAndPassword( const std::string & url ) {
    const std::string protoSeparator("://");

    size_type pos = url.find(protoSeparator);
    if( pos == std::string::npos ) {
        return 0;
    }

    pos += protoSeparator.length();

    const size_type endOfServerPos = url.find_first_of("/?#", pos);
    const size_type atPos = url.find('@', pos);

    if( atPos >= endOfServerPos  ) {
        return pos;
    }

    const std::string loginAndPassword = url.substr(pos, atPos-pos);
    const size_type semicolonPos = loginAndPassword.find(':');
    if( semicolonPos != std::string::npos ) {
        login_ = loginAndPassword.substr(0, semicolonPos);
        password_ = loginAndPassword.substr(semicolonPos+1);
    } else {
        login_ = loginAndPassword;
    }

    return atPos + 1;
}

url::size_type url::extractPort( const std::string & url, size_type pos) {
    const bool portMarkPresent(url[pos]==':');
    if( !portMarkPresent ) {
        return pos;
    }

    const size_type portPos = pos + 1;
    pos = url.find_first_of("/?#", portPos);
    if( pos != portPos ) {
        try {
            port( boost::lexical_cast<int>(url.substr(portPos, pos-portPos)));
        } catch ( const boost::bad_lexical_cast & ) {
            pos = portPos;
        }
    }
    return pos;
}

url::size_type url::extractServerAndPort( const std::string & url, size_type pos ) {
    size_type endPos = url.find_first_of("/?#:", pos);
    const size_type urlEndPos = endPos;

    if( endPos != std::string::npos ) {
        endPos = extractPort(url, endPos);
    }

    server_ = url.substr(pos, urlEndPos-pos);

    return endPos;
}

void url::extractUri( const std::string & url, size_type pos ) {
    if(pos != std::string::npos) {
        uri_ = url.substr(pos);
    }
}

void url::extractUrlPart( const std::string & url ) {
    size_type endPos = url.find_first_of("?#");
    urlPart_ = url.substr(0, endPos);
}

url::url(const std::string& s)
: protocol_(getUrlProto(s)), src_(s), uri_("/"),
  port_(getDefaultPort( protocol_ )), portSpecified_(false)
{
    size_type pos = extractLoginAndPassword(s);
    pos = extractServerAndPort(s, pos);
    extractUri( s, pos );
    extractUrlPart( s );
}

std::string url::protocol() const {
    switch(protocol_) {
        case UrlProto_HTTPS :
            return "https";
        case UrlProto_FTP :
            return "ftp";
        case UrlProto_HTTP :
            return "http";
        case UrlProto_OTHER :
        default: break;
    }
    return "";
}

url& url::relocate(const std::string& s)
{
    url n(s);
    if(!n.server().empty()) {
        server_ = n.server();
        port_ = n.port();
    }
    uri_ = n.uri();
    return *this;
}

int url::getDefaultPort( UrlProto protocol ) {
    switch(protocol) {
        case UrlProto_HTTPS :
            return 443;
        case UrlProto_FTP :
            return 21;
        case UrlProto_HTTP :
        case UrlProto_OTHER :
        default: break;
    }
    return 80;
}

} //namespace http
