#include <butil/network/rfc2822.h>

#include <recipient_parser/parse.hpp>

#include <boost/variant/get.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/transformed.hpp>

namespace rfc2822ns {
    using std::string;
    using std::list;
    typedef string::size_type size_type;

    bool atomchar(char c) {
        return !((c > 0  && c <= ' ') || c == '(' || c=='"' || c ==')'
                 || c == ',' || c == '.' || c == ':' || c == ';' || c == '<'
                 || c == '>' || c == '[' || c == ']' || c == '@');
    }

    string quote(const string& src)
    {
        bool quoted = false;
        string dst, space;
        dst.reserve(src.size() + 2);
        size_type i = 0;

        while(i < src.size() && src[i] == ' ') dst += src[i++];
        dst += '\"';

        for(; i < src.size(); ++i) {
            if(src[i] == ' ') space += src[i];
            else {
                dst += space;
                space.clear();
                if(src[i] == '\\' || src[i] == '\"') {
                    dst += '\\';
                    quoted = true;
                } else if(!atomchar(src[i]) && src[i] != '.')
                    quoted = true;
                dst += src[i];
            }
        }
        dst += '\"';
        dst += space;
        if(quoted) return dst;
        else return src;
    }

    string quote(const list<string>& src)
    {
        string dst;
        for(list<string>::const_iterator i = src.begin(); i != src.end(); ++i) {
            if((*i)[0] == '(') dst += " " + *i;
            else dst += quote(*i);
        }
        return dst;
    }

    string paste(const list<string>& src)
    {
        string dst;
        for(list<string>::const_iterator i = src.begin(); i != src.end(); ++i) {
            if((*i)[0] == '(') dst += " " + *i;
            else dst += *i;
        }
        return dst;
    }

    string join_address(const string& local, const string& domain) {
        if(domain.empty()) {
            return local;
        }
        return local + "@" + domain;
    }

    string join_address(const string& display,
                        const string& local, const string& domain) {
        if(display.empty()) return join_address(local, domain);
        return display + " <" + join_address(local, domain) + ">";
    }

    bool unquote_pair(const string& src, size_type& pos, string& dst)
    {
        if(pos + 1 >= src.size() || src[pos] != '\\') return false;
        pos++;
        dst += src[pos++];
        return true;
    }

    bool quoted_pair(const string& src, size_type& pos, string& dst)
    {
        if(pos + 1 >= src.size() || src[pos] != '\\') return false;
        dst += src[pos++];
        dst += src[pos++];
        return true;
    }

    void fws(const string& src, size_type& pos) {
        pos = src.find_first_not_of(" \t\r\n", pos);
        pos = std::min(pos, src.size());
    }

    bool fws(const string& src, size_type& pos, string& dst)
    {
        size_type tmppos = pos;
        fws(src, tmppos);
        if (tmppos == pos) {
            return false;
        }
        pos = tmppos;
        if (*dst.rbegin() != ' ') {
            dst += ' ';
        }
        return true;
    }

    bool comment(const string& src, size_type& pos, string& dst)
    {
        if(pos >= src.size() || src[pos] != '(') return false;
        size_type tmppos = pos;
        string tmpdst;
        int level = 0;
        tmpdst += src[tmppos++];
        while(tmppos < src.size() && (src[tmppos] != ')' || level > 0)) {
            if(src[tmppos] == '(') {
                level++;
            } else if(src[tmppos] == ')') {
                level--;
            }
            if(!quoted_pair(src, tmppos, tmpdst)
               && !fws(src, tmppos, tmpdst)) {
                tmpdst += src[tmppos++];
            }
        }
        if(src[tmppos] != ')' || level > 0) return false;
        tmpdst += src[tmppos++];
        dst += tmpdst;
        pos = tmppos;
        return true;
    }

    bool quoted_string(const string& src, size_type& pos, string& dst)
    {
        if(pos >= src.size() || src[pos] != '\"') return false;
        size_type tmppos = pos + 1;
        string tmpdst;
        while(tmppos < src.size() && src[tmppos] != '\"') {
            if(!unquote_pair(src, tmppos, tmpdst)
               && !fws(src, tmppos, tmpdst))
                tmpdst += src[tmppos++];
        }
        if(src[tmppos] != '\"') return false;
        dst += tmpdst;
        pos = tmppos + 1;
        return true;
    }



    bool atom(const string& src, size_type& pos, string& dst)
    {
        size_type tmppos = pos;
        string tmpdst;
        while(tmppos < src.size() && atomchar(src[tmppos]))
            tmpdst += src[tmppos++];
        if(src[tmppos] == '.') return false;
        if(tmppos == pos ||
           (tmppos < src.size() && src[tmppos] == '.'))
            return false;
        pos = tmppos;
        dst += tmpdst;
        return true;
    }

    bool dotatom(const string& src, size_type& pos, string& dst)
    {
        size_type tmppos = pos;
        string tmpdst;
        while(tmppos < src.size()
              && (atomchar(src[tmppos]) || src[tmppos] == '.'))
            tmpdst += src[tmppos++];
        if(tmppos == pos) return false;
        pos = tmppos;
        dst += tmpdst;
        return true;
    }

    bool domainDotAtom(const string& src, size_type& pos, string& dst)
    {
        size_type tmppos = pos;
        string tmpdst;
        while(tmppos < src.size() && src[tmppos] != '\''
              && (atomchar(src[tmppos]) || src[tmppos] == '.')) {
            tmpdst += src[tmppos++];
        }
        if(tmppos == pos || (tmppos < src.size() && src[tmppos] == '\'')) {
            return false;
        }
        pos = tmppos;
        dst += tmpdst;
        return true;
    }

    bool xatom(const string& src, size_type& pos, string& dst)
    {
        string tmpdst, space;
        size_type tmppos = pos;
        if(!dotatom(src, tmppos, tmpdst))
            return false;
        size_type xpos = tmppos;
        fws(src, xpos, space);
        if(src[xpos] == '@')
            return false;
        pos = tmppos;
        dst += tmpdst;
        return true;
    }

    bool display_name(const string& src, size_type& pos, list<string>& dst)
    {
        bool have_result = false;
        list<string> tmpdst;
        string data, commentdata;
        size_type tmppos = pos;
        while(tmppos < src.size()) {
            string space, tmpdata;
            fws(src, tmppos, space);
            if(!xatom(src, tmppos, tmpdata)
               && !quoted_string(src, tmppos, tmpdata)) {
                if(comment(src, tmppos, commentdata)) {
                    if(!data.empty()) tmpdst.push_back(data);
                    fws(src, tmppos, commentdata);
                    tmpdst.push_back(commentdata);
                    data.clear();
                    commentdata.clear();
                } else if(src[tmppos] == '\"' || src[tmppos] == '(')
                    data += space + src[tmppos++];
                else
                    break;
            } else if(tmpdata.empty()) {
                data += space;
                have_result = true;
            } else
                data += space + tmpdata;
        }
        if(!data.empty()) tmpdst.push_back(data);
        if(tmpdst.empty() && !have_result) return false;
        pos = tmppos;
        dst.swap(tmpdst);
        return true;
    }

    bool domain_literal(const string& src, size_type& pos, string& dst)
    {
        if(pos >= src.size() || src[pos] != '[') return false;
        size_type tmppos = pos;
        string tmpdst;
        tmpdst += src[tmppos++];
        while(tmppos < src.size() && src[tmppos] != ']') {
            if(src[tmppos] == '[') return false;
            if(!quoted_pair(src, tmppos, tmpdst)
               && !fws(src, tmppos, tmpdst))
               tmpdst += src[tmppos++];
        }
        if(src[tmppos] != ']') return false;
        tmpdst += src[tmppos++];
        dst += tmpdst;
        pos = tmppos;
        return true;
    }

    bool addrspec(const string& src, size_type& pos,
                  string& local, string& domain)
    {
        string tmplocal, tmpdomain, space;
        string::size_type tmppos = pos;
        if(!dotatom(src, tmppos, tmplocal)
           && !quoted_string(src, tmppos, tmplocal))
            return false;
        fws(src, tmppos, space);
        if(src[tmppos] != '@') return false;
        tmppos++;
        fws(src, tmppos, space);
        if(!domainDotAtom(src, tmppos, tmpdomain)
           && !domain_literal(src, tmppos, tmpdomain))
            return false;
        pos = tmppos;
        local += tmplocal;
        domain += tmpdomain;
        return true;
    }

    bool addrspec(const string& src, size_type& pos, string& dst)
    {
        string local, domain;
        if(!addrspec(src, pos, local, domain)) return false;
        dst += join_address(local, domain);
        return true;
    }

    bool angleaddr(const string& src, size_type& pos,
                   string& local, string& domain)
    {
        string tmplocal, tmpdomain;
        size_type tmppos = pos;
        if(tmppos >= src.size() || src[tmppos] != '<') return false;
        tmppos++;
        if(!addrspec(src, tmppos, tmplocal, tmpdomain)
           && !dotatom(src, tmppos, tmplocal))
            return false;
        if(tmppos >= src.size() || src[tmppos] != '>') return false;
        tmppos++;
        pos = tmppos;
        local += tmplocal;
        domain += tmpdomain;
        return true;
    }

    bool angleaddr(const string& src, size_type pos, string& dst)
    {
        string local, domain;
        if(!angleaddr(src, pos, local, domain)) return false;
        dst += join_address(local, domain);
        return true;
    }

    size_type skip_end_group(const string& src, size_type pos) {
        const size_t found = src.find_first_not_of(",;:", pos);
        return std::min(found, src.size());
    }

    size_type skip_group(const string& src, size_type start_pos) {
        size_t pos = src.find_first_of("\"<;,:", start_pos);
        if (pos == string::npos) {
            return start_pos;
        }
        if (src[pos] == ':') {
            ++pos;
            fws(src, pos);
            return skip_end_group(src, pos);
        } else {
            return start_pos;
        }
    }

    bool address(const string& src, size_type& pos,
                 list<string>& display, string& local, string& domain)
    {
        list<string> tmpdisplay;
        string tmplocal, tmpdomain, space;
        size_type tmppos = pos;
        tmppos = skip_group(src, tmppos);
        fws(src, tmppos, space);
        if (tmppos == src.size()) {
            pos = tmppos;
            return false;
        }
        bool have_name = display_name(src, tmppos, tmpdisplay);
        fws(src, tmppos, space);
        if(!angleaddr(src, tmppos, tmplocal, tmpdomain)
           && !addrspec(src, tmppos, tmplocal, tmpdomain))
            if(!have_name) return addrspec(src, pos, local, domain);
        pos = tmppos;
        display.swap(tmpdisplay);
        local = tmplocal;
        domain = tmpdomain;
        return true;
    }

    bool address(const string& src, size_type& pos,
                 string& display, string& addr)
    {
        list<string> tmpdisplay;
        string local, domain;
        if(!address(src, pos, tmpdisplay, local, domain)) return false;
        display += quote(tmpdisplay);
        addr = join_address(local, domain);
        return true;
    }

    old_address_iterator& old_address_iterator::operator++(void) {
        list<string> tmpdisplay;
        string space, tmplocal, tmpdomain;
        rfc2822ns::fws(src_, pos_, space);
        while(pos_ < src_.size()
              && (src_[pos_] == ',' || src_[pos_] == ';')) {
            pos_++;
            rfc2822ns::fws(src_, pos_, space);
        }
        if(!rfc2822ns::address(src_, pos_, tmpdisplay, tmplocal, tmpdomain)) {
            if(pos_ < src_.size()) {
                string text = src_.substr(pos_, src_.size() - pos_);
                throw invalid_address(text);
            }
            pos_ = string::npos;
        } else {
            local_ = tmplocal;
            domain_ = tmpdomain;
            display_ = quote(tmpdisplay);
            pretty_ = paste(tmpdisplay);
            address_ = join_address(local_, domain_);
        }
        return *this;
    }

    const string old_address_iterator::empty;



new_address_iterator::new_address_iterator() {
    current = recipients.begin();
}

new_address_iterator::new_address_iterator(Recipients&& rcpts)
    : recipients(std::move(rcpts)), current(recipients.begin()) {
}

std::optional<new_address_iterator> new_address_iterator::parse(const std::string& src) {
    auto iter = src.begin();
    const auto end = src.end();

    auto result = rcpt_parser::parse_address_list(iter, end);
    if (!result || iter != end) {
        return std::nullopt;
    }

    auto recipients = rcpt_parser::flat_mailbox_list(*result);
    return new_address_iterator{std::move(recipients)};
}

bool new_address_iterator::operator == (const new_address_iterator& other) {
    const bool pointOnSameRcpt =
            recipients == other.recipients
         && current    == other.current;
    const bool bothExhausted =
                  current ==       recipients.end()
         && other.current == other.recipients.end();
    return pointOnSameRcpt || bothExhausted;
}

bool new_address_iterator::operator != (const new_address_iterator& other) {
    return !(*this == other);
}

new_address_iterator& new_address_iterator::operator ++(void) {
    ++current;
    return *this;
}

string new_address_iterator::display(void) const {
    auto quoted = boost::adaptors::transformed(
            [](const std::string& word){ return quote(word); }
    );
    return boost::algorithm::join(current->display_name | quoted, " ");
}

string new_address_iterator::pretty(void) const {
    return boost::algorithm::join(current->display_name, " ");
}

const string& new_address_iterator::local(void) const {
    return current->addr_spec.login;
}

const string& new_address_iterator::domain(void) const {
    return current->addr_spec.domain;
}

string new_address_iterator::address(void) const {
    return join_address(local(), domain());
}



address_iterator::address_iterator() {}

address_iterator::address_iterator(const std::string& src) {
    if (auto new_addr_iter = new_address_iterator::parse(src)) {
        impl = std::make_shared<address_iterator_impl<new_address_iterator>>(std::move(*new_addr_iter));
    } else {
        impl = std::make_shared<address_iterator_impl<old_address_iterator>>(src);
    }
}

bool address_iterator::operator == (const address_iterator& other) {
    return (impl == other.impl) || (this->empty() && other.empty());
}

address_iterator& address_iterator::operator ++() {
    impl->advance();
    return *this;
}

string address_iterator::display() const {
    return impl->display();
}

string address_iterator::pretty() const {
    return impl->pretty();
}

string address_iterator::local() const {
    return impl->local();
}

string address_iterator::domain() const {
    return impl->domain();
}

string address_iterator::address() const {
    return impl->address();
}


}
