from __future__ import absolute_import, print_function

"""
Processor for calendar attaches to submit calendar events
"""

import logging

from itertools import imap
from os import environ

from email.parser import HeaderParser
from psycopg2 import Error

from calendar_attach_processor.arc import is_arcadia
from calendar_attach_processor.service.blackbox import BlackboxService, BlackboxError, User
from calendar_attach_processor.service.calendar import Envelope, CalendarEvent, Calendar, CalendarError
from calendar_attach_processor.service.mdb import MailPg, MailPgError
from calendar_attach_processor.service.ml import Maillist, MlError
from calendar_attach_processor.service.sharpei import Sharpei, SharpeiError
from calendar_attach_processor.service.storage import Storage, StorageError
from calendar_attach_processor.tvm.session import TvmSession

log = logging.getLogger(__name__)

X_CALENDAR_ENV = "X-Calendar-Env"
X_CALENDAR_DOMAIN = "X-Calendar-Domain"
X_CALENDAR_REQ_ID = "X-Calendar-Request-Id"


def recipient2email(rcpt):
    if not rcpt.local and not rcpt.domain:
        return None
    return "{}@{}".format(rcpt.local, rcpt.domain)


def parse_address(s):
    # type: (str) -> Optional[str]
    if is_arcadia():
        import address_parser
        return recipient2email(address_parser.parse_recipients(s)[0])
    else:
        from flanker.addresslib import address
        parsed = address.parse(s)
        return parsed


def parse_address_list(s):
    # type: (str) -> List[str]
    if not s:
        return []
    if is_arcadia():
        import address_parser
        return [recipient2email(rcpt) for rcpt in address_parser.parse_recipients(s)]
    else:
        from flanker.addresslib import address
        return [email.address for email in address.parse_list(s)]


class CalendarMailhookProcessor(object):
    @classmethod
    def from_config(cls, cfg, domain, env):
        return cls(
            blackbox=BlackboxService(cfg['blackbox']['url'], self_ip=cfg['blackbox']['self_ip']),
            sharpei=Sharpei(cfg['sharpei'], cfg['pg']['user'], environ[cfg['pg']['pwd_env']]),
            storage=Storage(cfg["storage"]),
            ml=Maillist(cfg['ml']['url'],
                        TvmSession(cfg['ml']['tvm']['client'],
                                   environ[cfg['ml']['tvm']['secret_env']])),
            calendar=Calendar(cfg['calendar']),
            domain=domain,
            env=env,
        )

    @staticmethod
    def get_envelope(headers):
        """
        Parse "from", "cc", "message-id" headers and create an envelope
        Content of this headers is needed to submit event in calendar

        :param headers: email headers to parse
        :type headers: email.message.Message
        """

        parsed_from = parse_address(headers.get('from'))
        message_id = headers.get('message-id')
        if not parsed_from:
            log.info("service=calendar\toper=parse_from\tstatus=failed\terr=wrong from <%s>", headers.get('from'))
            address_from = None
        else:
            address_from = parsed_from
        try:
            cc = parse_address_list(headers.get('cc'))
            return Envelope(address_from, cc, message_id)
        except (KeyError, AttributeError):
            log.info("service=calendar\toper=parse_cc\tstatus=failed\terr=wrong cc <%s>", headers.get('cc'))
            return Envelope(address_from, [], message_id)

    def __init__(self, blackbox, sharpei, storage, ml, calendar, domain, env):
        if env is None:
            raise TypeError("The 'env' argument should be of 'str' type, but 'None' is passed")
        self.blackbox = blackbox
        self.sharpei = sharpei
        self.storage = storage
        self.ml = ml
        self.calendar = calendar
        self.domain = domain
        self.env = env

    def process_calendar_attach(self, uid, mid, stid, hid, content_type):
        """
        Process calendar attach and submit event in calendar if needed

        :param uid: user id in passport
        :param mid: message id
        :param stid: storage id
        :param hid: id of mime part to process
        :param content_type: content-type of mime part to process
        :param config: service config
        :return: True - if calendar event was submitted, otherwise - False
        """

        try:
            # Check user info in blackbox
            user = self.blackbox.resolve_recipient_by_uid(uid)

            if not user.is_pg:
                log.debug("service=calendar-mailhook\tuid=%s\tmid=%s\tstatus=skip_not_pg_user", uid, mid)
                return False

            # Get mime parts of message from database
            dsn = self.sharpei.get_conn_info(uid)
            mime_parts = MailPg(dsn).get_mime(uid, mid)

            if not any(mime.hid == hid for mime in mime_parts):
                log.error("service=calendar-mailhook\tuid=%s\tmid=%s\tstatus=failed_to_get_mime_parts", uid, mid)
                return False

            raw_headers = self.storage.stid(stid, mime_parts).headers()
            headers = HeaderParser().parsestr(raw_headers)

            if X_CALENDAR_REQ_ID in headers:
                headers_env = headers.get(X_CALENDAR_ENV)
                # If headers contain X-Calendar-Request-Id, X-Calendar-Env and X-Calendar-Domain
                # and domain + env equals to the current domain + current environment,
                # it means event has been submitted already. There is no need to submit it again
                if headers.get(X_CALENDAR_DOMAIN) == self.domain and headers_env == self.env:
                    log.debug("service=calendar-mailhook\tuid=%s\tmid=%s\tstatus=has_req_id_and_same_domain+env", uid, mid)
                    return False

                if headers_env != self.env:
                    log.debug("service=calendar-mailhook\tuid=%s\tmid=%s\tstatus=has_different_env", uid, mid)
                    return False

            users = []
            if user.is_ml and self.domain == 'yt':
                # If user is maillist, request subscribers from ml and
                # filter them by excluding ones with subscription in inbox
                email = user.login + "@yandex-team.ru"
                subscribers = filter(lambda subscriber: not subscriber.inbox,
                                     self.ml.expand_maillists(email))
                # If resulted subscribers list is empty, return False as there is no one to submit event for
                if len(subscribers) == 0:
                    log.debug("service=calendar-mailhook\tuid=%s\temail=%s\tstatus=ml_list_has_no_emails",
                              uid, email)
                    return False
                # For each subscriber get user info in passport
                logins = imap(lambda subscriber: subscriber.email, subscribers)
                users = self.blackbox.resolve_uids_by_logins(logins)
            else:
                to = ', '.join(parse_address_list(headers.get('to')))
                users.append(User(uid=user.uid, login=user.login, email=to))

            # temporary hack for testing without testing environment
            if self.env == 'production':
                # Now we have all necessary information.
                # Create event and submit it in calendar
                envelope = CalendarMailhookProcessor.get_envelope(headers)
                attach = self.storage.stid(stid, mime_parts).attach(hid)
                event = CalendarEvent(users, envelope, attach)

                self.calendar.mailhook(event, mid, hid, stid)

            log.debug("service=calendar_attach_processor"
                      "\tuid=%s\tmid=%s\tdomain=%s\tenv=%s\tml=%s\tstid=%s\thid=%s\ttype=%s\tstatus=submit",
                      uid, mid, self.domain, self.env, user.is_ml, stid, hid, content_type)

            return True

        except BlackboxError as err:
            log.error("service=blackbox\tdomain=%s\tenv=%s\tuid=%s\tstatus=failed\terr=%s", self.domain, self.env, uid, err)
        except SharpeiError as err:
            log.error("service=sharpei\toper=get_conn_info\tdomain=%s\tenv=%s\tuid=%s\tstatus=failed\terr=%s",
                      self.domain, self.env, uid, err)
        except StorageError as err:
            log.error("service=storage\toper=get\tdomain=%s\tenv=%s\tuid=%s\tmid=%s\tstid=%s\thid=%s\tstatus=failed\terr=%s",
                      self.domain, self.env, uid, mid, stid, hid, err)
        except MlError as err:
            log.error("service=ml\toper=expand_maillists\tuid=%s\tstatus=failed\terr=%s", uid, err)
        except CalendarError as err:
            log.error("service=calendar\toper=mailhook\tdomain=%s\tenv=%s\tuid=%s\tmid=%s\tstatus=failed\terr=%s",
                      self.domain, self.env, uid, mid, err)
        except (Error, MailPgError) as err:
            log.error("service=mail_pg\toper=get_mime\tdomain=%s\tenv=%s\tuid=%s\tmid=%s\tstatus=failed\terr=%s",
                      self.domain, self.env, uid, mid, err)

        return False
