import logging
from collections import namedtuple

from blackboxer import Blackbox, HTTPError, TransportError, ResponseError

DB_ID_DBFIELD = "hosts.db_id.2"
ACCOUNT_IS_MAILLIST_ATTRIBUTE = "13"

User = namedtuple('User', ('uid', 'login', 'is_ml', 'email', 'is_pg'))
User.__new__.__defaults__ = (None, None, False, None, True)


class BlackboxError(Exception):
    pass


class BlackboxService(object):
    def __init__(self, url, self_ip, timeout=10):
        self.blackbox = Blackbox(url, timeout=timeout)
        self.timeout = timeout
        self.self_ip = self_ip

    def resolve_recipient_by_uid(self, uid):
        return self._userinfo(uid=uid, attributes=ACCOUNT_IS_MAILLIST_ATTRIBUTE, dbfields=DB_ID_DBFIELD)

    def resolve_uids_by_logins(self, logins):
        for login in logins:
            try:
                yield self._userinfo(login=login)
            except BlackboxError as e:
                logging.error("service=calendar-mailhook\tlogin=%s\tstatus=userinfo_fail\terr=%s", login, e)

    def _userinfo(self, uid=None, login=None, **extra):
        try:
            userinfo = self.blackbox.userinfo(userip=self.self_ip, uid=uid, login=login, **extra)
            return self.extract_user(userinfo, email=login)
        except (HTTPError, TransportError, ResponseError) as err:
            raise BlackboxError(err)

    @staticmethod
    def extract_user(userinfo, email=None):
        try:
            found_user = next(user for user in userinfo.get('users', ())
                              if 'login' in user and 'uid' in user)

            login = found_user['login']
            uid = found_user['uid']['value']
            is_ml = found_user.get('attributes', {}).get(ACCOUNT_IS_MAILLIST_ATTRIBUTE, "0") == "1"
            is_pg = found_user.get('dbfields', {}).get(DB_ID_DBFIELD, "") == "pg"

            return User(uid, login=login, is_ml=is_ml, email=email, is_pg=is_pg)
        except ValueError as err:
            raise BlackboxError(err)
        except StopIteration:
            raise BlackboxError("invalid user")
