from collections import namedtuple
from concurrent.futures import ThreadPoolExecutor

import ticket_parser2.api.v1 as tvm

from .utils import retry_session

Attach = namedtuple('Attach', ('content_type', 'content'))
Envelope = namedtuple('Envelope', ('from_email', 'cc', 'message_id'))
CalendarEvent = namedtuple('CalendarEvent', ('users', "envelope", 'attach'))


class CalendarError(Exception):
    pass


class Calendar(object):
    def __init__(self, url, session=retry_session(), timeout=10):
        self.url = url
        self.session = session
        self.timeout = timeout
        self.pool = ThreadPoolExecutor(max_workers=25)
        self._tvm_client = None

    @property
    def tvm_client(self):
        # Lazy initialization is used because Calendar instance is instantiated in legacy logbroker-client
        # in parent process before spawning worker Process instances,
        # so ticket_parser2 throws an exception "Creating TvmClient is forbidden before fork".
        # Laziness allows us to instantiate TvmClient in the worker process.
        # If started as a single process worker, laziness would not be needed.
        if self._tvm_client is None:
            self._tvm_client = tvm.TvmClient(tvm.TvmToolClientSettings('attach-processor', port=1))
        return self._tvm_client

    def mailhook(self, calendar_event, mid=None, hid=None, stid=None):
        def post(user):
            return self.session.post(
                self.url,
                params={
                    'content-type': calendar_event.attach.content_type,
                    'message-id': calendar_event.envelope.message_id,
                    'yandex-uid': user.uid,
                    'to': user.email,
                    'cc': ','.join(calendar_event.envelope.cc),
                    'from': calendar_event.envelope.from_email,
                    'mid': mid,
                    'hid': hid,
                    'stid': stid,
                },
                headers={
                    'Content-Type': calendar_event.attach.content_type,
                    'X-Ya-Service-Ticket': service_ticket,
                },
                data=calendar_event.attach.content,
                timeout=self.timeout,
            )

        service_ticket = self.tvm_client.get_service_ticket_for('calendar')
        submissions = self.pool.map(post, calendar_event.users)

        if all(s.status_code != 200 for s in submissions):
            raise CalendarError("calendar request failed for all uids")
