import logging

import psycopg2
import psycopg2.extras
from psycopg2 import Error
from retry import retry

logging.basicConfig(level=logging.DEBUG)


class MailPgError(Exception):
    pass


class LoggingCursor(psycopg2.extensions.cursor):
    def execute(self, sql, args=None):
        try:
            psycopg2.extensions.cursor.execute(self, sql, args)
        except Error:
            logging.warning(self.mogrify(sql, args))
            raise


class MailPg(object):
    def __init__(self, dsn):
        self.dsn = dsn

    @retry(Error, tries=3, delay=1, jitter=(1, 2))
    def get_mime(self, uid, mid):
        """
        More info:
        - http://initd.org/psycopg/docs/extras.html#composite-types-casting
        - https://github.yandex-team.ru/mail/mdb/blob/master/mail.sql

        :param uid:
        :param mid:
        :return:
        """
        sql_mime = """SELECT mime::mail.mime_part[] FROM mail.messages WHERE  uid = %s AND  mid = %s LIMIT 1"""

        with psycopg2.connect(self.dsn) as conn:
            logging.debug('calendar-mailhook: pg_conn=%s' % conn)
            psycopg2.extras.register_composite('mail.mime_part', conn)
            with conn.cursor(cursor_factory=LoggingCursor) as curs:
                curs.execute(sql_mime, (uid, mid))
                mime = curs.fetchone()
                if mime is None or len(mime) == 0:
                    return []
                return mime[0]
