import requests
import logging
import base64
import quopri

from .utils import http_request, retry_session
from .calendar import Attach

STORAGE_REQ_FORMAT = "{url}/gate/get/{stid}"
log = logging.getLogger(__name__)


class StorageError(Exception):
    pass


class Storage:
    def __init__(self, url):
        self.url = url

    def stid(self, stid, mime_parts):
        url = STORAGE_REQ_FORMAT.format(url=self.url, stid=stid)

        return MDSRequest(url, mime_parts)


class StorageRequest(object):
    def __init__(self, mime_parts):
        self.mime_parts = mime_parts

    @staticmethod
    def find_mime(mime_parts, hid):
        """
         Used for requesting a mime part of an email from storage

        :param mime_parts: mime parts to search in
        :param hid: id of requested mime part
        :return: the offsets of requested mime part
        """

        for mime_part in mime_parts:
            if mime_part.hid == hid:
                return mime_part
        raise StorageError("can't find mime part with hid={0}".format(hid))

    @staticmethod
    def decode_base64(content):
        return base64.b64decode(content)

    @staticmethod
    def decode_qp(content, charset):
        return quopri.decodestring(content).decode(charset)

    def headers(self):
        pass

    def content(self, hid):
        pass

    def attach(self, hid):
        attach = StorageRequest.find_mime(self.mime_parts, hid)

        try:
            content = {
                'base64': lambda c: StorageRequest.decode_base64(c),
                'quoted-printable': lambda c: StorageRequest.decode_qp(c, attach.charset),
                '7bit': lambda c: c,
                '8bit': lambda c: c,
                'binary': lambda c: c,
            }[attach.encoding.lower()](self.content(hid))

            return Attach('/'.join((attach.content_type, attach.content_subtype)), content)
        except KeyError as e:
            raise StorageError('unknown encoding: {}'.format(e.args[0]))
        except Exception as e:
            raise StorageError('decode failed for {}: {}'.format(attach.encoding, e))


class MDSRequest(StorageRequest):
    def __init__(self, url, mime_parts, session=retry_session(), timeout=10):
        super(MDSRequest, self).__init__(mime_parts)
        self.url = url
        self.session = session
        self.timeout = timeout

    def headers(self):
        content_begin = StorageRequest.find_mime(self.mime_parts, '1').offset_begin

        storage_params = {'service': 'calendar-mailhook'}
        range_header = {'Range': "bytes={begin}-{end}".format(begin=0, end=content_begin)}

        try:
            response = http_request(self.url, session=self.session, params=storage_params, headers=range_header,
                                    timeout=self.timeout)
            if response.status_code != 206:
                raise StorageError("wrong mds status={0}, expecting 206".format(response.status_code))
            return response.content
        except requests.HTTPError as e:
            raise StorageError(e)

    def content(self, hid):
        content = StorageRequest.find_mime(self.mime_parts, hid)

        storage_params = {'service': 'calendar-mailhook'}
        range_header = {'Range': "bytes={begin}-{end}".format(begin=content.offset_begin, end=content.offset_end)}

        try:
            response = http_request(self.url, session=self.session, params=storage_params, headers=range_header,
                                    timeout=self.timeout)
            if response.status_code != 206:
                raise StorageError("wrong mds status={0} for hid={1}, expecting 206".format(response.status_code, hid))
            return response.content
        except requests.HTTPError as e:
            raise StorageError(e)


# // TODO: after full mds migration mulca type can be removed completely
# Mulca sends some metainfo in the begining. To force it stop this - we should send gettype param
# But MDS will say that this request is wrong with 400 status. So we should retry
# After migration can be removed
class MulcaRequest(StorageRequest):
    def __init__(self, url, mime_parts, session=retry_session(), timeout=10):
        super(MulcaRequest, self).__init__(mime_parts)
        self.url = url
        self.session = session
        self.timeout = timeout

    def headers(self):
        storage_params = {
            'service': 'calendar-mailhook',
            'gettype': 'meta'
        }

        try:
            return http_request(self.url, session=self.session, params=storage_params, timeout=self.timeout).content
        except requests.HTTPError as e:
            log.debug("service=storage\ttype=mulca\trequest=headers\terror=%s", e)
            # fallback
            return MDSRequest(self.url, self.mime_parts).headers()

    def content(self, hid):
        storage_params = {
            'service': 'calendar-mailhook',
            'gettype': 'part',
            'part': hid
        }

        try:
            return http_request(self.url, session=self.session, params=storage_params, timeout=self.timeout).content
        except requests.HTTPError as e:
            log.debug("service=storage\ttype=mulca\trequest=content\thid=%s\terror=%s", hid, e)
            # fallback
            return MDSRequest(self.url, self.mime_parts).content(hid)
