# encoding: utf-8
from __future__ import unicode_literals

import requests
import cachetools

from logging import getLogger

from utvm import TVM, TvmException

from requests.adapters import HTTPAdapter
from requests.exceptions import RetryError, ConnectionError, ConnectTimeout, SSLError, ReadTimeout, Timeout
from requests.status_codes import _codes
from urllib3 import Retry

# Http status codes for retry
RETRY_STATUS_CODES = filter(lambda code: code >= 500, _codes.keys())

# A backoff factor to apply between attempts after the second try. If the backoff_factor is 0.1,
# then :func:`.sleep` will sleep for [0.0s, 0.2s, 0.4s, ...] between retries.
BACKOFF_FACTOR = 0.2

logger = getLogger(__name__)

TVMError = TvmException


class TvmSession(requests.Session):
    cache = cachetools.TTLCache(128, 1800)

    @staticmethod
    def tvm_from(client_id, secret):
        return TVM(client_id=client_id,
                   secret=secret,
                   allowed_v1_clients=(client_id,),
                   use_v1=False)

    def __init__(self, client_id, secret, retries_num=2, status_forcelist=RETRY_STATUS_CODES, *args, **kwargs):
        super(TvmSession, self).__init__(*args, **kwargs)
        self._dst = client_id
        self._tvm = TvmSession.tvm_from(client_id, secret)

        retries = Retry(total=retries_num,
                        backoff_factor=BACKOFF_FACTOR,
                        status_forcelist=status_forcelist)

        self.mount('http://', HTTPAdapter(max_retries=retries))
        self.mount('https://', HTTPAdapter(max_retries=retries))

    @cachetools.cached(cache=cache)
    def get_ticket_headers(self):
        return self._tvm.ticket_getter().get_service_ticket_headers(self._dst)[str(self._dst)]

    def request(self, *args, **kwargs):
        headers = kwargs.pop('headers', None)
        tvm_headers = self.get_ticket_headers()
        if headers:
            for k, v in tvm_headers.items():
                headers[k] = v
        else:
            headers = tvm_headers
        try:
            response = super(TvmSession, self).request(*args, headers=headers, **kwargs)
            response.raise_for_status()
            return response
        except (ConnectionError, ReadTimeout, ConnectTimeout, Timeout, SSLError, SSLError, RetryError) as err:
            raise requests.HTTPError(err)
        except TVMError:
            raise requests.HTTPError(response='tvm failed')
