#!/usr/bin/python3
# -*- coding: UTF-8 -*-
import os

from alive_app_pb2 import AliveApp
from collections import defaultdict
from kazoo.client import KazooClient
from bottle import route, run, template
import qloud
import re
import json
import requests



GROUP_TO_HOSTS_URL = "https://c.yandex-team.ru/api/groups2hosts/"
TEMPLATE_PATH = "/app/admin/main.tpl"


def get_alive_apps(zk_hosts, zk_path, project):
    alive_apps = []
    zk = KazooClient(hosts=",".join(zk_hosts), timeout=2.0)
    zk.start()

    for node in zk.get_children(zk_path):
        app = AliveApp()
        app.ParseFromString(zk.get(zk_path + node)[0])

        alive_apps.append({"name": app.appName, "host": app.hostname, "version": app.version})

    zk.stop()
    return alive_apps

def get_bazinga_master(zk_hosts, zk_path, project):
    zk = KazooClient(hosts=",".join(zk_hosts), timeout=2.0)
    zk.start()
    current_master_node = min(zk.get_children(zk_path))
    master = zk.get(zk_path + current_master_node)[0]
    master = master.decode('utf-8')
    master_host = master.split(":")[0]
    zk.stop()
    return master_host

def is_qloud(host):
    if re.search("qloud-c.yandex.net", host):
        return True
    return False


def cname_iaas(host, app_name, admin_project):
    host = re.sub("cmail.yandex.net", "yandex-team.ru", host)
    return ".".join([admin_project + "-" + app_name, host])


def cname_qloud(component_name, app_name, qloud_application, qloud_environment):
    return ".".join([component_name, app_name, qloud_environment, qloud_application, "yandex-team.ru"])


def get_zk_hosts(group):
    return [host["fqdn"] for host in json.loads(requests.get(GROUP_TO_HOSTS_URL + group + "?format=json").text)]


def get_qloud_instances_names(qloud_environment):
    qloud_instances_names = {}
    for instnace in qloud.get_qloud_instances(qloud_environment, name=True):
        qloud_instances_names[instnace["fqdn"]] = instnace["name"]
    return qloud_instances_names

def get_app_info():

    zk_path = qloud.environ.user["ZK_PATH"] if not qloud.DEPLOY_STAGE_ID else os.getenv("ZK_PATH")
    zk_group = qloud.environ.user["ZK_GROUP"] if not qloud.DEPLOY_STAGE_ID else os.getenv("ZK_GROUP")
    zookeeper_hosts = get_zk_hosts(zk_group)
    admin_project = zk_path.split("/")[1]

    bazinga_master = get_bazinga_master(zookeeper_hosts, zk_path + "/bazinga/election/", admin_project)
    zk_apps = get_alive_apps(zookeeper_hosts,
                         zk_path + "/alive-apps/", admin_project)

    qloud_instances_names = get_qloud_instances_names(qloud.application)

    app_info = defaultdict(list)
    for app in zk_apps:
        component_name = "iaas"
        is_master = False

        if is_qloud(app["host"]):
            instance_name = qloud_instances_names.get(app["host"], "unknown")
            component_name = "-".join(instance_name.split("-")[:-1])
            cname = cname_qloud(instance_name, app["name"], qloud.qloud_application, qloud.environment)
        else:
            cname = cname_iaas(app["host"], app["name"], admin_project)

        if app["host"] == bazinga_master:
            is_master = True

        app_info[app["name"]].append(
            {"host": app["host"],
            "component": component_name,
            "cname": cname,
            "version": app["version"],
            "is_master": is_master})
    return app_info


@route('/')
@route('/admin')
def root():
    app_info = get_app_info()
    return template(TEMPLATE_PATH,
        app_info=app_info,
        qloud_application=qloud.qloud_application,
        qloud_environment_name=qloud.environment)


@route('/ping')
def ping():
    return 'pong'


run(host='::', port=8080, debug=True)
