package ru.yandex.calendar.generate;

import lombok.val;
import org.apache.commons.codec.Charsets;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;


public class DownloadTanker {
    public static class DownloadSource {
        String source;
        List<String> languages;

        DownloadSource(String source, List<String> lang) {
            this.source = source;
            this.languages = lang;
        }
    }

    private static final String BASEURL = "https://tanker-api.yandex-team.ru/tools/generate/i18n.xml";

    private static String postProccess(String content) {
        content = content.replaceAll("<xsl:apply-templates mode=\"lego:copy\" select=\"i18n:node-set\\(([^)]*)\\)\"/>",
                "<xsl:copy-of select=\"$1\"/>");
        content = content.replaceAll("<xsl:apply-templates select=\"i18n:node-set\\(([^)]*)\\)\" mode=\"lego:copy\"/>",
                "<xsl:copy-of select=\"$1\"/>");
        content = content.replaceAll("xmlns:x=\"http://www.yandex.ru/xscript\"\\s+select=\"x:tolower\\(([^)]*)\\)\"",
                "select=\"lower-case($1)\"");
        return content;
    }

    private static boolean downloadKeyset(String language, String project, String source, String outputDir) {
        val url = String.format("%s?language=%s&project-id=%s&keyset-id=%s&type=xsl&status=unapproved",
                BASEURL, language, project, source);

        val request = new HttpGet(url);

        try (val httpClient = HttpClients.createDefault()) {
            val res = httpClient.execute(request);
            if (res.getStatusLine().getStatusCode() != 200) {
                return false;
            } else {
                String content = EntityUtils.toString(res.getEntity(), Charsets.UTF_8);
                content = postProccess(content);
                val outputFileName = Paths.get(outputDir, language, project + "." + source + ".xsl").toString();
                val file = new File(outputFileName);

                if (!file.getParentFile().exists() && !file.getParentFile().mkdirs()) {
                    System.err.println("Can't find file: " + file.toString());
                    return false;
                } else {
                    try (val writer = new PrintWriter(file.toString(), Charsets.UTF_8)) {
                        writer.print(content);
                    }
                }
            }
        } catch (IOException e) {
            System.err.println(e.toString());
            return false;
        }
        return true;
    }

    public static void main(String[] args) {
        val outputDir = args[0];

        val emails = new DownloadSource("emails2", Arrays.asList("ru", "en"));
        val todoEmails = new DownloadSource("todo-email", Arrays.asList("ru", "az", "be", "en", "hy", "ka", "kk", "ro", "tr", "tt", "uk"));
        val days = new DownloadSource("days", Arrays.asList("ru", "en"));
        val dynamic = new DownloadSource("dynamic", Arrays.asList("ru", "en"));
        val months = new DownloadSource("months", Arrays.asList("ru", "en"));

        val downloadStaff = new HashMap<String, List<DownloadSource>>();
        downloadStaff.put("calendar", Arrays.asList(emails, todoEmails));
        downloadStaff.put("tanker", Arrays.asList(days, dynamic, months));

        boolean[] successFlag = {true};
        downloadStaff.entrySet().stream().forEach(
                prj -> prj.getValue().stream().forEach(
                        keyset -> {
                            keyset.languages.stream().forEach(
                                    language -> {
                                        if (successFlag[0] && !downloadKeyset(language, prj.getKey(), keyset.source, outputDir)) {
                                           successFlag[0] = false;
                                        }
                                    }
                            );
                        }
                )
        );
        if (!successFlag[0]) {
            System.err.println("Can't get sources, some of download sources failed");
            System.exit(1);
        }
    }
}
