package ru.yandex.calendar.generate;

import com.google.common.base.Strings;
import lombok.val;
import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.text.WordUtils;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

public class Bean {
    private final Pattern namePattern = Pattern.compile("CREATE TABLE\\s+(\\S+)");
    private final int MAX_ID_FIELDS_IN_TABLE = 3;
    private String content;
    private String tableName;
    private List<Field> fields;

    public Bean(String tableContent) {
        content = tableContent;
    }

    void parseTable() {
        val m = namePattern.matcher(content);
        if (m.find()) {
            tableName = m.group(1);
        } else {
            throw new RuntimeException("Can't extract table name for " + content);
        }
        fields = Arrays.stream(content.split("\n"))
                .filter(line -> line.matches(Field.fieldPattern.toString()))
                .map(Field::new).collect(Collectors.toList());
    }

    private boolean isNeedOption() {
        return fields.stream().anyMatch(Field::isNullable);
    }

    private boolean isNeedNullableEnum() {
        return fields.stream().anyMatch(Field::isNullableEnum);
    }

    String getBeanClassName() {
        return Arrays.stream(tableName.split("_")).map(WordUtils::capitalize).collect(Collectors.joining(""));
    }

    String getBeanName() {
        return tableName;
    }

    List<Field> getFields() {
        return fields;
    }

    private List<Field> getIdFields() {
        return fields.stream().filter(Field::getIsId).collect(Collectors.toList());
    }

    private int getIdFieldsCount() {
        val res = (int) fields.stream().filter(Field::getIsId).count();
        if (res > MAX_ID_FIELDS_IN_TABLE) {
            throw new RuntimeException("Unexpected id fields size");
        }
        return res;
    }

    List<String> getImports(boolean isBeanDef, List<String> defaultImports) {
        val imports = new ArrayList<String>(defaultImports);
        if (isNeedOption() && isBeanDef) {
            imports.add("ru.yandex.bolts.collection.Option");
        }
        if (isNeedNullableEnum()) {
            imports.add("ru.yandex.calendar.logic.beans.NullableOtherValue");
            if (!isBeanDef) {
                imports.add("ru.yandex.calendar.logic.beans.NullableOtherValueType");
                imports.add("ru.yandex.commune.mapObject.db.type.StringEnumValueType");
            }
        }
        if (isBeanDef) {
            val res = getIdContainerClass();
            res.ifPresent(imports::add);
        }
        fields.stream().map(Field::getType).filter(type -> type.contains(".")).forEach(imports::add);
        if (!isBeanDef) {
            fields.stream().map(f -> f.dbValueType.orElse("")).filter(type -> type.contains(".")).forEach(imports::add);
        }
        return imports;
    }

    List<String> getBeanHelperImports(List<String> beanHelperImports) {
        val imports = getIdFields().stream().filter(f -> f.getType().contains(".")).map(Field::getType).collect(Collectors.toList());
        val idContainerClass = getIdContainerClass();
        idContainerClass.ifPresent(imports::add);
        imports.addAll(beanHelperImports);
        return imports;
    }

    private Optional<String> getIdContainerClass() {
        switch (getIdFieldsCount()) {
            case 2:
                return Optional.of("ru.yandex.bolts.collection.Tuple2");
            case 3:
                return Optional.of("ru.yandex.bolts.collection.Tuple3");
            default:
                return Optional.empty();
        }
    }

    String getIdType() {
        val s = fields.stream().filter(Field::getIsId).map(Field::getSimpleType).collect(Collectors.joining(", "));
        switch (getIdFieldsCount()) {
            case 1:
                return s;
            case 2:
                return String.format("Tuple2<%s>", s);
            case 3:
                return String.format("Tuple3<%s>", s);
        }
        return "";
    }

    private String getValueGetter(Field f) {
        return String.format("getFieldValue(%sFields.%s)", getBeanClassName(), f.getName().toUpperCase());
    }

    private String getValueSetter(Field f, int num) {
        String getN = num != 0 ? String.format(".get(%d)", num) : "";
        val template = new StringTemplate("setFieldValue($beanClassName$Fields.$fieldName$, id$getN$)");
        template.setAttribute("beanClassName", getBeanClassName());
        template.setAttribute("fieldName", f.getName().toUpperCase());
        template.setAttribute("getN", getN);
        return template.toString();
    }

    String getBeanIdGetter() {
        val idFields = getIdFields();
        if (idFields.size() == 1) {
            StringTemplate template = new StringTemplate("        return $valueGetter$;");
            template.setAttribute("valueGetter", getValueGetter(idFields.get(0)));
            return template.toString();
        } else if (idFields.size() == 2) {
            val getterStr = idFields.stream().map(f -> Strings.repeat(" ", 16) + getValueGetter(f)).collect(Collectors.joining(",\n"));
            return Strings.repeat(" ", 8) + String.format("return Tuple2.tuple(\n%s);", getterStr);
        } else if (idFields.size() == 3) {
            val getterStr = idFields.stream().map(f -> Strings.repeat(" ", 16) + getValueGetter(f)).collect(Collectors.joining(",\n"));
            return Strings.repeat(" ", 8) + String.format("return Tuple3.tuple(\n%s);", getterStr);
        }
        return "";
    }

    String getBeanIdSetter() {
        val idFields = getIdFields();
        if (idFields.size() == 1) {
            return Strings.repeat(" ", 8) + getValueSetter(idFields.get(0), 0) + ";";
        } else {
            return IntStream.range(0, idFields.size())
                    .mapToObj(i -> Strings.repeat(" ", 8) + getValueSetter(idFields.get(i), i + 1) + ";")
                    .collect(Collectors.joining("\n"));
        }
    }

    void printBeanToFile(File outputDir) throws FileNotFoundException, UnsupportedEncodingException {
        val outputFile = Paths.get(outputDir.toString(), getBeanClassName() + ".java").toString();
        try (val writer = new PrintWriter(outputFile, "UTF-8")) {
            writer.print(Printer.generateBeanString(this));
        }
    }

    void printBeanFieldsToFile(File outputDir) throws FileNotFoundException, UnsupportedEncodingException {
        val outputFile = Paths.get(outputDir.toString(), getBeanClassName() + "Fields.java").toString();
        try (val writer = new PrintWriter(outputFile, "UTF-8")) {
            writer.print(Printer.generateBeanFieldsString(this));
        }
    }

    void printBeanHelperToFile(File outputDir) throws FileNotFoundException, UnsupportedEncodingException {
        val outputFile = Paths.get(outputDir.toString(), getBeanClassName() + "Helper.java").toString();
        try (val writer = new PrintWriter(outputFile, "UTF-8")) {
            writer.print(Printer.generateBeanHelperString(this));
        }
    }
}
