package ru.yandex.calendar.generate;

import lombok.val;
import org.apache.commons.text.WordUtils;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static java.util.Collections.singletonList;

public class Field {
    static final Pattern fieldPattern = Pattern.compile("^\\s+([a-z]\\w+)\\s+(\\w+)(\\[\\])?(?:\\((\\d+)\\))?(.+?)(-- .+)?$");
    private static final Pattern javaTypePattern = Pattern.compile("JAVA_TYPE=(\\S+)");
    private final Map<String, List<String>> javaTypeToSqlTypes = Map.of(
        "Integer", List.of("TINYINT", "SMALLINT", "MEDIUMINT", "INT", "INTEGER"),
        "Long", List.of("BIGINT", "BIGSERIAL"),
        "Boolean", List.of("BIT", "BOOL", "BOOLEAN"),
        "Double", List.of("FLOAT", "DOUBLE", "DECIMAL"),
        "String", List.of("CHAR", "VARCHAR", "ENUM", "SET", "TEXT", "BLOB", "JSONB"),
        "org.joda.time.LocalTime", singletonList("TIME"),
        "org.joda.time.LocalDate", List.of("DATE", "YEAR"),
        "org.joda.time.Instant", List.of("DATETIME", "TIMESTAMP")
    );
    private final Map<String, String> typeOverrideToDbValueType = Map.of(
        "UUID", "ru.yandex.calendar.logic.beans.UuidValueType",
        "MessageParameters", "ru.yandex.calendar.logic.beans.MailMessageValueType",
        "MailerRepetition", "ru.yandex.calendar.logic.beans.MailerRepetitionValueType",
        "MailerParticipant", "ru.yandex.calendar.logic.beans.MailerParticipantValueType",
        "MailerAttendees", "ru.yandex.calendar.logic.beans.MailerAttendeesValueType",
        "IntegerArray", "ru.yandex.calendar.logic.beans.IntegerArrayType",
        "RawJson", "ru.yandex.calendar.frontend.bender.RawJsonValueType"
    );
    private final List<String> overrideTypes = List.of(
            "ru.yandex.calendar.frontend.ews.sync.IgnoreReason",
            "ru.yandex.calendar.logic.event.ActionSource",
            "ru.yandex.calendar.logic.event.avail.Availability",
            "ru.yandex.calendar.logic.event.grid.ViewType",
            "ru.yandex.calendar.logic.event.model.EventType",
            "ru.yandex.calendar.logic.event.model.Completion",
            "ru.yandex.calendar.logic.event.model.Priority",
            "ru.yandex.calendar.logic.event.repetition.RegularRepetitionRule",
            "ru.yandex.calendar.logic.layer.LayerType",
            "ru.yandex.calendar.logic.resource.ResourceType",
            "ru.yandex.calendar.logic.sharing.Decision",
            "ru.yandex.calendar.logic.sharing.InvAcceptingType",
            "ru.yandex.calendar.logic.sharing.perm.EventActionClass",
            "ru.yandex.calendar.logic.sharing.perm.LayerActionClass",
            "ru.yandex.calendar.logic.todo.TodoStatus",
            "ru.yandex.calendar.logic.todo.TodoViewType",
            "ru.yandex.calendar.util.dates.DayOfWeek",
            "ru.yandex.inside.passport.PassportSid",
            "ru.yandex.inside.passport.PassportUid",
            "ru.yandex.misc.email.Email",
            "ru.yandex.calendar.logic.user.Language",
            "ru.yandex.calendar.logic.notification.Channel",
            "java.util.UUID",
            "ru.yandex.calendar.logic.sending.param.MessageParameters",
            "ru.yandex.calendar.logic.mailer.model.MailerRepetition",
            "ru.yandex.calendar.logic.mailer.model.MailerParticipant",
            "ru.yandex.calendar.logic.mailer.model.MailerAttendees",
            "ru.yandex.calendar.logic.beans.IntegerArray",
            "ru.yandex.calendar.frontend.bender.RawJson"
    );
    Optional<String> dbValueType;
    private String line, type, name;
    private int typeLength;
    private boolean isId;
    private boolean isNullable;
    private boolean isNullableEnum;
    private boolean isCut;
    private boolean isNotEmpty;

    public Field(String str) {
        line = str;
        val m = fieldPattern.matcher(line);
        if (m.find()) {
            val fieldName = m.group(1);
            val sqlType = m.group(2);
            val typeLength = m.group(4);
            val tail = m.group(5);
            val comment = m.group(6);

            dbValueType = Optional.empty();
            name = fieldName;
            isId = fieldName.equals("id") || (comment != null && comment.contains("_ID"));
            isNullable = tail == null || !tail.contains("NOT NULL");
            isNullableEnum = comment != null && comment.contains("NULLABLE_ENUM");
            isCut = comment != null && comment.contains("CUT");
            isNotEmpty = comment != null && comment.contains("NOT EMPTY");
            if (isCut && sqlType.equalsIgnoreCase("text")) {
                this.typeLength = 4000;
            } else {
                this.typeLength = typeLength != null ? Integer.parseInt(typeLength) : 0;
            }
            fillTypes(comment == null ? "" : comment, sqlType);
        } else {
            throw new RuntimeException("Can't parse line as field to field: " + line);
        }
    }

    boolean getIsId() {
        return isId;
    }

    String getType() {
        return type;
    }

    String getName() {
        return name;
    }

    private void fillTypes(String comment, String sqlType) {
        val m = javaTypePattern.matcher(comment);
        if (m.find()) {
            val overrideType = m.group(1);
            if (typeOverrideToDbValueType.containsKey(overrideType)) {
                dbValueType = Optional.of(typeOverrideToDbValueType.get(overrideType));
            }

            type = overrideTypes
                .stream()
                .filter(t -> getWordAfterLastDot(t).equals(overrideType))
                .findFirst()
                .orElseThrow(() -> new RuntimeException("Unknown override java type: " + overrideType + " for line: " + line));
        } else {
            type = javaTypeToSqlTypes.entrySet()
                .stream()
                .filter(kv -> kv.getValue().contains(sqlType))
                .map(Map.Entry::getKey)
                .findFirst()
                .orElseThrow(() -> new RuntimeException("Unknown sql type: " + sqlType + " for line: " + line));
        }
    }

    String getNameUpperCase() {
        return name.toUpperCase();
    }

    String getNameCamelCase() {
        return Arrays.stream(name.split("_")).map(WordUtils::capitalize).collect(Collectors.joining(""));
    }

    boolean isNullable() {
        return isNullable;
    }

    boolean isNullableEnum() {
        return isNullableEnum;
    }

    Optional<String> getDbValueType() {
        return dbValueType;
    }

    int getTypeLength() {
        return typeLength;
    }

    boolean getIsCut() {
        return isCut;
    }

    boolean getIsNotEmpty() {
        return isNotEmpty;
    }

    String getWordAfterLastDot(String s) {
        return s.substring(s.lastIndexOf('.') + 1);
    }

    String getSimpleType() {
        if (type.contains(".")) {
            return getWordAfterLastDot(type);
        }
        return type;
    }

    String getPrimitiveType() {
        String res = type;

        if (type.equals("Integer")) {
            res = "int";
        }
        if (res.contains(".")) {
            res = getWordAfterLastDot(res);
        } else if (!res.equals("String")) {
            res = res.substring(0, 1).toLowerCase() + res.substring(1);
        }
        return res;
    }

    String getSimpleDbValueType() {
        final var res = dbValueType.map(dbType -> dbType.contains(".") ?  getWordAfterLastDot(dbType) : dbType);
        return res.orElseThrow(() -> new RuntimeException("Can't get simple dbvalue type for field: " + line));
    }
}
