package ru.yandex.calendar.generate;

import java.util.List;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import lombok.val;
import org.antlr.stringtemplate.StringTemplate;

class Printer {
    private final static List<String> beanImports = List.of(
            "ru.yandex.calendar.logic.beans.Bean",
            "ru.yandex.calendar.logic.beans.BeanHelper",
            "ru.yandex.commune.mapObject.MapObjectDescription",
            "ru.yandex.bolts.function.Function");
    private final static List<String> beanFieldsImports = List.of(
            "ru.yandex.commune.mapObject.MapField",
            "ru.yandex.commune.mapObject.MapObjectBuilder",
            "ru.yandex.commune.mapObject.MapObjectDescription"
    );
    private final static List<String> beanHelperImports = List.of(
            "ru.yandex.calendar.logic.beans.BeanHelper",
            "ru.yandex.commune.mapObject.MapObjectDescription"
    );

    static String prepareImports(List<String> imports) {
        return imports.stream().sorted().distinct().map(s -> "import " + s + ";").collect(Collectors.joining("\n"));
    }

    static String generateFieldStringForBeanHelper(Field field) {
        String result = "";
        if (field.getIsId()) {
            result = "id()";
        } else {
            result = "b()";
            if (field.getIsNotEmpty()) {
                result += ".notEmpty()";
            } else if (field.isNullableEnum()) {
                result += ".notNull()\n";
                result += Strings.repeat(" ", 12);
                result += ".type(NullableOtherValue.class)\n";
                result += Strings.repeat(" ", 12);
                result += String.format(".dbValueType(new NullableOtherValueType<>(new StringEnumValueType<>(%s.R)))\n", field.getSimpleType());
                result += Strings.repeat(" ", 12);
            } else if (!field.isNullable()) {
                result += ".notNull()";
            }
            if (field.getDbValueType().isPresent()) {
                result += ".dbValueType(new " + field.getSimpleDbValueType() + "())";
            }
            if (field.getIsCut()) {
                result += String.format(".cut(%s)", field.getTypeLength());
            }
            result += ".f()";
        }
        val template = new StringTemplate("\n    MapField<$fieldType$> $fieldUpperName$ = MapField.$f$;");
        template.setAttribute("fieldType", field.isNullableEnum() ? String.format("NullableOtherValue<%s>", field.getSimpleType()) : field.getSimpleType());
        template.setAttribute("fieldUpperName", field.getNameUpperCase());
        template.setAttribute("f", result);
        return template.toString();
    }


    static String generateFieldStringForBean(String beanClassName, Field field) {
        if (field.getName().equals("id")) {
            return "";
        }
        if (field.isNullableEnum()) {
            val template = new StringTemplate("\n\n" +
                    "    public Option<$fieldType$> get$fieldCamelName$() {\n" +
                    "        return getFieldValue($beanClassName$Fields.$fieldUpperName$).asOption();\n" +
                    "    }\n" +
                    "\n" +
                    "    public void set$fieldCamelName$($fieldType$ value) {\n" +
                    "        setFieldValue($beanClassName$Fields.$fieldUpperName$, value == null\n" +
                    "                ? NullableOtherValue.none()\n" +
                    "                : NullableOtherValue.some(value));\n" +
                    "    }\n" +
                    "\n" +
                    "    public void set$fieldCamelName$Null() {\n" +
                    "        setFieldValue($beanClassName$Fields.$fieldUpperName$, NullableOtherValue.none());\n" +
                    "    }\n" +
                    "\n" +
                    "    public void set$fieldCamelName$(Option<$fieldType$> value) {\n" +
                    "        setFieldValue($beanClassName$Fields.$fieldUpperName$, new NullableOtherValue<>(value));\n" +
                    "    }\n" +
                    "\n" +
                    "    public static Function<$beanClassName$, Option<$fieldType$>> get$fieldCamelName$F() {\n" +
                    "        return $beanClassName$::get$fieldCamelName$;\n" +
                    "    }");
            template.setAttribute("fieldType", field.getSimpleType());
            template.setAttribute("beanClassName", beanClassName);
            template.setAttribute("fieldUpperName", field.getNameUpperCase());
            template.setAttribute("fieldCamelName", field.getNameCamelCase());
            return template.toString();
        } else if (field.isNullable()) {
            val template = new StringTemplate("\n\n" +
                    "    public Option<$fieldType$> get$fieldCamelName$() {\n" +
                    "        return getFieldValueNullAsNone($beanClassName$Fields.$fieldUpperName$);\n" +
                    "    }\n" +
                    "\n" +
                    "    public void set$fieldCamelName$($fieldType$ value) {\n" +
                    "        setFieldValue($beanClassName$Fields.$fieldUpperName$, value);\n" +
                    "    }\n" +
                    "\n" +
                    "    public void set$fieldCamelName$Null() {\n" +
                    "        setFieldValue($beanClassName$Fields.$fieldUpperName$, null);\n" +
                    "    }\n" +
                    "\n" +
                    "    public void set$fieldCamelName$(Option<$fieldType$> value) {\n" +
                    "        setFieldValue($beanClassName$Fields.$fieldUpperName$, value.getOrNull());\n" +
                    "    }\n" +
                    "\n" +
                    "    public static Function<$beanClassName$, Option<$fieldType$>> get$fieldCamelName$F() {\n" +
                    "        return $beanClassName$::get$fieldCamelName$;\n" +
                    "    }");
            template.setAttribute("fieldType", field.getSimpleType());
            template.setAttribute("beanClassName", beanClassName);
            template.setAttribute("fieldUpperName", field.getNameUpperCase());
            template.setAttribute("fieldCamelName", field.getNameCamelCase());
            return template.toString();
        } else {
            val template = new StringTemplate("\n\n" +
                    "    public $fieldPrimitiveType$ get$fieldCamelName$() {\n" +
                    "        return getFieldValue($beanClassName$Fields.$fieldUpperName$);\n" +
                    "    }\n" +
                    "\n" +
                    "    public void set$fieldCamelName$($fieldPrimitiveType$ value) {\n" +
                    "        setFieldValue($beanClassName$Fields.$fieldUpperName$, value);\n" +
                    "    }\n" +
                    "\n" +
                    "    public static Function<$beanClassName$, $fieldType$> get$fieldCamelName$F() {\n" +
                    "        return $beanClassName$::get$fieldCamelName$;\n" +
                    "    }");
            template.setAttribute("fieldType", field.getSimpleType());
            template.setAttribute("fieldPrimitiveType", field.getPrimitiveType());
            template.setAttribute("beanClassName", beanClassName);
            template.setAttribute("fieldUpperName", field.getNameUpperCase());
            template.setAttribute("fieldCamelName", field.getNameCamelCase());
            return template.toString();
        }
    }

    static String generateBeanString(Bean bean) {
        val template = new StringTemplate("package ru.yandex.calendar.logic.beans.generated;\n" +
                "\n" +
                "$imports$\n" +
                "\n" +
                "public class $beanClassName$ extends Bean<$beanIdType$> {\n" +
                "\n" +
                "    @Override\n" +
                "    public $beanClassName$ clone() {\n" +
                "        return ($beanClassName$) super.clone();\n" +
                "    }\n" +
                "\n" +
                "    @Override\n" +
                "    public BeanHelper<$beanClassName$, $beanIdType$> getHelper() {\n" +
                "        return $beanClassName$Helper.INSTANCE;\n" +
                "    }\n" +
                "\n" +
                "    @Override\n" +
                "    public MapObjectDescription getMapObjectDescription() {\n" +
                "        return $beanClassName$Fields.OBJECT_DESCRIPTION;\n" +
                "    }\n" +
                "\n" +
                "    @Override\n" +
                "    public $beanIdType$ getId() {\n" +
                "$beanIdGetter$\n" +
                "    }\n" +
                "\n" +
                "    @Override\n" +
                "    public void setId($beanIdType$ id) {\n" +
                "$beanIdSetter$\n" +
                "    }\n" +
                "\n" +
                "    public static Function<$beanClassName$, $beanIdType$> getIdF() {\n" +
                "        return $beanClassName$::getId;\n" +
                "    }");
        template.setAttribute("imports", prepareImports(bean.getImports(true, beanImports)));
        template.setAttribute("beanClassName", bean.getBeanClassName());
        template.setAttribute("beanIdType", bean.getIdType());
        template.setAttribute("beanIdGetter", bean.getBeanIdGetter());
        template.setAttribute("beanIdSetter", bean.getBeanIdSetter());
        val endTemplate = new StringTemplate("\n\n" +
                "} // $beanClassName$\n");
        endTemplate.setAttribute("beanClassName", bean.getBeanClassName());
        return template + bean.getFields().stream().map(f -> Printer.generateFieldStringForBean(bean.getBeanClassName(), f)).collect(Collectors.joining()) + endTemplate.toString();
    }

    static String generateBeanFieldsString(Bean bean) {
        val template = new StringTemplate("package ru.yandex.calendar.logic.beans.generated;\n" +
                "\n" +
                "$imports$\n" +
                "\n" +
                "public interface $beanClassName$Fields {\n");
        template.setAttribute("imports", prepareImports(bean.getImports(false, beanFieldsImports)));
        template.setAttribute("beanClassName", bean.getBeanClassName());

        val endTemplate = new StringTemplate("\n\n" +
                "    MapObjectDescription OBJECT_DESCRIPTION =\n" +
                "        MapObjectBuilder.describe(\"$beanTableName$\", $beanClassName$Fields.class);\n" +
                "\n" +
                "} // $beanClassName$Fields\n");
        endTemplate.setAttribute("beanTableName", bean.getBeanName());
        endTemplate.setAttribute("beanClassName", bean.getBeanClassName());
        return template + bean.getFields().stream().map(Printer::generateFieldStringForBeanHelper).collect(Collectors.joining("\n")) + endTemplate.toString();
    }

    static String generateBeanHelperString(Bean bean) {
        val template = new StringTemplate("package ru.yandex.calendar.logic.beans.generated;\n" +
                "\n" +
                "$imports$\n" +
                "\n" +
                "public class $beanClassName$Helper extends BeanHelper<$beanClassName$, $beanIdType$> {\n" +
                "    public static final $beanClassName$Helper INSTANCE = new $beanClassName$Helper();\n" +
                "\n" +
                "    @Override\n" +
                "    public MapObjectDescription beanMapObjectDescription() {\n" +
                "        return $beanClassName$Fields.OBJECT_DESCRIPTION;\n" +
                "    }\n" +
                "} // $beanClassName$\n");
        template.setAttribute("imports", prepareImports(bean.getBeanHelperImports(beanHelperImports)));
        template.setAttribute("beanClassName", bean.getBeanClassName());
        template.setAttribute("beanIdType", bean.getIdType());
        return template.toString();
    }
}
