package ru.yandex.calendar;

import java.util.OptionalLong;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.misc.cache.tl.TlCache;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.thread.WhatThreadDoes;

@Slf4j
public class CalendarRequest {
    public static CalendarRequestHandle push(ActionSource actionSource, String what) {
        return push(actionSource, what, "");
    }

    public static CalendarRequestHandle push(ActionSource actionSource, String what, String actionForLog) {
        return push(new RemoteInfo(Option.empty(), Option.empty()), actionSource, what, actionForLog);
    }

    public static CalendarRequestHandle push(RemoteInfo remoteInfo, ActionSource actionSource, String what, String actionForLog) {
        return push(remoteInfo, actionSource, what, actionForLog, Instant.now(), false);
    }

    public static CalendarRequestHandle push(RemoteInfo remoteInfo, ActionSource actionSource, String what,
                                             String actionForLog, Instant now, boolean freezeNowForTest) {
        return push(remoteInfo, actionSource, what, actionForLog, now, freezeNowForTest, OptionalLong.empty());
    }

    public static CalendarRequestHandle push(RemoteInfo remoteInfo, ActionSource actionSource, String what,
                                             String actionForLog, Instant now, boolean freezeNowForTest, OptionalLong requestTvmId) {
        val requestIdStackHandle = RequestIdStack.pushIfNotYet();
        val tlCacheHandle = TlCache.push();
        val wtdHandle = WhatThreadDoes.push(what);
        log.info(String.format("remoteInfo = %s, actionSource = %s, what = %s, actionForLog = %s, now = %s, " +
                "freezeNowForTest = %s, requestTvmId = %s", remoteInfo, actionSource, what, actionForLog, now,
                freezeNowForTest, requestTvmId));
        log.info("{}", LogMarker.ACTION_SOURCE.format(actionSource.toDbValue()));

        val prev = CalendarRequestHandle.current.get();
        val actionInfo = new ActionInfo(actionForLog, actionSource, requestIdStackHandle.getValue(), now, freezeNowForTest, requestTvmId);
        val current = new CalendarRequestHandle(prev, requestIdStackHandle, tlCacheHandle, wtdHandle, remoteInfo, actionInfo);

        CalendarRequestHandle.current.set(current);
        return current;
    }

    public static CalendarRequestHandle push(CalendarRequestHandle request) {
        CalendarRequestHandle.current.set(request);
        return request;
    }

    public static Option<CalendarRequestHandle> getCurrentO() {
        return Option.ofNullable(CalendarRequestHandle.current.get());
    }

    public static Option<String> getCurrentConnectionId() {
        return getCurrentO().flatMapO(h -> h.getRemoteInfo().connectionId);
    }

    public static CalendarRequestHandle getCurrent() {
        return CalendarRequestHandle.current.get();
    }
}
