package ru.yandex.calendar.admin;

import lombok.val;
import org.dom4j.Element;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.idlent.YandexUser;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.env.Environment;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.reflection.JavaBeanProperty;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;


public class BeanXmlizerForAdmin {
    private static final Logger logger = LoggerFactory.getLogger(BeanXmlizerForAdmin.class);

    @Autowired
    private UserManager userManager;
    @Autowired
    private ResourceDao resourceDao;

    public Element xmlize(final Object any) {
        return xmlize(any, Option.empty());
    }

    public Element xmlize(Object any, Option<Either<SettingsInfo, ResourceInfo>> participant) {
        Element r = Dom4jUtils.createElement(CamelWords.parse(any.getClass().getSimpleName()).toXmlName());
        for (JavaBeanProperty getter : ClassX.getClass(any).getProperties()) {
            CamelWords words = CamelWords.parse(getter.getName());
            String elementName = words.toXmlName();

            Object value = getter.get(any);

            if (value instanceof Option<?> && ((Option<?>) value).isPresent()) {
                value = ((Option<?>) value).get();
            }

            Option<PassportUid> uidO = Option.empty();
            Option<String> loginO = Option.empty();
            Option<String> emailO = Option.empty();

            if (words.getWords().last().equals("uid")) {
                if (value instanceof Long) {
                    uidO = Option.of(PassportUid.cons((Long) value));
                } else if (value instanceof PassportUid) {
                    uidO = Option.of((PassportUid) value);
                }
                if (participant.exists(Either::isLeft)) {
                    loginO = participant.get().getLeft().getUserLogin();
                    emailO = Option.of(Emails.getUnicoded(participant.get().getLeft().getEmail()));

                } else if (uidO.isPresent() && !Environment.isDeveloperNotebook()) {
                    try {
                        val ytUser = userManager.getYtUserByUid(uidO.get());
                        if (ytUser.isPresent()) {
                            loginO = Option.of(ytUser.get().getLogin());
                            emailO = StringUtils.notEmptyO(ytUser.get().getInfo().getWorkEmail());
                        } else {
                            YandexUser user = userManager.getUserByUid(uidO.get()).getOrThrow("not found ", uidO.get());
                            loginO = Option.of(user.getLogin().getRawValue());
                            emailO = user.getUnicodedEmail();
                        }
                    } catch (Exception e) {
                        logger.warn(e, e);
                    }
                }
                value = Cf.list(value).plus(emailO).mkString(" ");
            }

            Option<Resource> resourceO = Option.empty();

            if (elementName.equals("resource-id")) {
                long id = (long) value;

                if (participant.exists(Either::isRight)) {
                    resourceO = Option.of(participant.get().getRight().getResource());
                } else {
                    resourceO = Option.of(resourceDao.findResourceById(id));
                }
                value = Cf.list(value,
                        resourceO.get().getName().getOrElse(""),
                        ResourceRoutines.getResourceEmail(resourceO.get())
                ).mkString(" ");
            }

            // exchange ids are typically 168 chars - it's good to see them entirely to be able to copy
            String valueText = StringUtils.take(StringUtils.lines(ObjectUtils.toString(value)).mkString("\\n"), 170, "...");

            Element element = Dom4jUtils.createElementWithContent(elementName, valueText);
            if (uidO.isPresent()) {
                element.addAttribute("uid", uidO.get().toString());
                if (loginO.isPresent()) {
                    element.addAttribute("login", loginO.get());
                }
                if (emailO.isPresent()) {
                    element.addAttribute("email", emailO.get());
                }
            }
            if (resourceO.isPresent()) {
                element.addAttribute("resource-id", Long.toString(resourceO.get().getId()));
                element.addAttribute("name", resourceO.get().getName().getOrElse(""));
                element.addAttribute("exchange-name", resourceO.get().getExchangeName().getOrElse(""));
                element.addAttribute("email", ResourceRoutines.getResourceEmail(resourceO.get()).toString());
            }

            if (elementName.equals("exchange-id") && !(value instanceof Option.None)) {
                element.addAttribute("exchange-id", UrlUtils.urlEncode(valueText));
            }
            r.add(element);
        }
        return r;
    }

} //~
