package ru.yandex.calendar.admin;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.logic.LastUpdateManager;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.LayerUser;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventLayerId;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.layer.LayerDao;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerType;
import ru.yandex.calendar.logic.layer.LayerUserDao;
import ru.yandex.calendar.logic.notification.NotificationDbManager;
import ru.yandex.calendar.logic.sharing.perm.LayerActionClass;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;

@Slf4j
public class GiftManager {
    @Autowired
    private EventDao eventDao;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private LayerDao layerDao;
    @Autowired
    private LayerUserDao layerUserDao;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private NotificationDbManager notificationDbManager;
    @Autowired
    private LastUpdateManager lastUpdateManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private TransactionTemplate transactionTemplate;


    public void giveMeetingToUserByEventIdAndEmail(long eventId, Email taker) {
        Option<PassportUid> uid = userManager.getUidByEmail(taker);
        Validate.some(uid, "User not found by email " + taker);

        giveMeetingToUserByEventIdAndUid(eventId, uid.get());
    }

    public void giveMeetingToUserByEventIdAndUid(long eventId, PassportUid taker) {
        Option<PassportUid> organizer = eventUserDao.findOrganizerByEventId(eventId);
        Validate.some(organizer, "Event not found, have no organizer or organizer is not user");
        Validate.isFalse(organizer.get().sameAs(taker), "Already is organizer");

        long mainEventId = eventDao.findEventById(eventId).getMainEventId();
        ListF<Long> eventIds = eventDao.findEventIdsByMainEventId(mainEventId);

        giveMeetingsToUser(eventIds, organizer.get(), taker);
        log.info("Event organizer replaced successfully");
    }

    public void giveCalendarToUserByLayerIdAndEmail(long layerId, Email taker) {
        val uid = userManager.getUidByEmail(taker).getOrThrow("User not found by email " + taker);
        giveCalendarToUserByLayerIdAndUid(layerId, uid);
    }

    public void giveCalendarToUserByLayerIdAndUid(long layerId, PassportUid taker) {
        val layer = layerDao.findLayerByIdSafe(layerId).getOrThrow("Layer not found");
        if (layer.getType() != LayerType.USER) {
            throw new RuntimeException("Layer type is not USER");
        }
        if (layer.getCreatorUid().equalsTs(taker)) {
            throw new RuntimeException("Already is owner");
        }
        giveCalendarToUser(layer.getId(), layer.getCreatorUid(), taker);
        log.info("Layer and all events owner changed successfully");
    }

    private void giveMeetingsToUser(final ListF<Long> eventIds, final PassportUid owner, final PassportUid taker) {
        transactionTemplate.execute(status -> {
            val takerLayerId = layerRoutines.getOrCreateDefaultLayer(taker);
            eventDao.lockEventsByIds(eventIds);

            replaceEventUsersUidAndEventsCreatorUid(eventIds, owner, taker);
            replaceEventLayersLayerIdAndCreatorUid(eventIds, owner, taker, takerLayerId);
            return null;
        });
    }

    private void giveCalendarToUser(final long layerId, final PassportUid owner, final PassportUid taker) {
        transactionTemplate.execute(status -> {
            val eventIds = eventDao.lockEventsByLayerId(layerId);

            replaceLayerUserUidAndLayerCreatorUid(layerId, taker);
            replaceEventUsersUidAndEventsCreatorUid(eventIds, owner, taker);
            replaceEventLayersLayerIdAndCreatorUid(eventIds, owner, taker, layerId);
            settingsRoutines.updateDefaultLayer(owner, null);
            return null;
        });
    }

    private void replaceLayerUserUidAndLayerCreatorUid(long layerId, PassportUid taker) {
        val actionInfo = ActionInfo.adminManager();
        val owner = layerDao.findLayerById(layerId).getCreatorUid();

        val ownerLayerUserId = layerUserDao.findLayerUserByLayerIdAndUid(layerId, owner).map(LayerUser::getId);
        val takerLayerUserId = layerUserDao.findLayerUserByLayerIdAndUid(layerId, taker).map(LayerUser::getId);

        if (ownerLayerUserId.isPresent() && takerLayerUserId.isPresent()) {
            notificationDbManager.deleteNotificationsByLayerUserIds(ownerLayerUserId);
            layerUserDao.deleteLayerUsersByIds(ownerLayerUserId);
            layerUserDao.updateLayerUserPerm(layerId, taker, LayerActionClass.ADMIN);
        } else if (ownerLayerUserId.isPresent()) {
            layerUserDao.updateLayerUsersSetUidByIds(ownerLayerUserId, taker);
        }
        layerDao.updateLayerSetCreatorUidById(layerId, taker, actionInfo);
    }

    private void replaceEventUsersUidAndEventsCreatorUid(ListF<Long> eventIds, PassportUid owner, PassportUid taker) {
        Validate.notEquals(owner, taker);
        final ActionInfo actionInfo = ActionInfo.adminManager();

        ListF<EventUser> ownerEventUsers = eventUserDao.findEventUsersByEventIdsAndUid(eventIds, owner);
        ListF<EventUser> takerEventUsers = eventUserDao.findEventUsersByEventIdsAndUid(eventIds, taker);

        MapF<Long, EventUser> takerEventUserByEventId = takerEventUsers.toMapMappingToKey(EventUser.getEventIdF());

        Tuple2<ListF<EventUser>, ListF<EventUser>> t = ownerEventUsers
                .partition(EventUser.getEventIdF().andThen(takerEventUserByEventId::containsKeyTs));

        ListF<Long> eventUserIdsToRemove = t.get1().map(EventUser.getIdF());
        ListF<Long> eventUserIdsToChangeUid = t.get2().map(EventUser.getIdF());
        ListF<Long> eventUserIdsToSetOrganizer = t.get1().filter(EventUser::getIsOrganizer)
                .map(eu -> takerEventUserByEventId.getOrThrow(eu.getEventId()).getId());

        notificationDbManager.deleteNotificationsByEventUserIds(eventUserIdsToRemove);
        eventUserDao.deleteEventUsersByIds(eventUserIdsToRemove);
        eventUserDao.updateEventUsersSetOwnerUidByIds(eventUserIdsToChangeUid, taker, actionInfo);
        eventUserDao.updateEventUsersSetIsOrganizerAndIsAttendeeByIds(eventUserIdsToSetOrganizer, true, true, actionInfo);

        ListF<Long> eventIdsToChangeCreator = eventDao.findEventsByIdsSafe(eventIds)
                .filter(Event.getCreatorUidF().andThenEquals(owner)).map(Event.getIdF());
        eventDao.updateEventSetCreatorUidByIds(eventIdsToChangeCreator, taker, actionInfo);

        lastUpdateManager.updateMainEventAndLayerTimestamps(ownerEventUsers.map(EventUser.getEventIdF()), actionInfo);
    }

    private void replaceEventLayersLayerIdAndCreatorUid(
            ListF<Long> eventIds, PassportUid owner, PassportUid taker, long targetLayerId)
    {
        Validate.notEquals(owner, taker);
        final ActionInfo actionInfo = ActionInfo.adminManager();

        ListF<EventLayer> ownerEventLayers = eventLayerDao.findEventLayersByLayerIdAndEventIds(targetLayerId, eventIds);
        ListF<EventLayer> takerEventLayers = eventLayerDao.findEventLayersByEventIdsAndLayerCreatorUid(eventIds, taker);

        SetF<Long> commonEventIds = ownerEventLayers.map(EventLayer::getEventId).unique()
                .intersect(takerEventLayers.map(EventLayer::getEventId).unique());

        eventLayerDao.deleteEventLayersByIds(takerEventLayers
                .filterMap(el -> Option.when(commonEventIds.containsTs(el.getEventId()), EventLayerId.of(el))));

        ownerEventLayers.map(EventLayerId::of).paginate(1000)
                .forEach(eventLayerIds -> eventLayerDao.updateEventLayersSetLayerIdAndLayerCreatorUidByIds(
                        eventLayerIds, targetLayerId, taker, actionInfo));
    }
}
