package ru.yandex.calendar.admin.displays;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.admin.z.ZRedirectException;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;

/**
 * @author dbrylev
 */
@ActionContainer
public class DisplaysAdminPage {

    private final ResourceRoutines resourceRoutines;

    public DisplaysAdminPage(ResourceRoutines resourceRoutines) {
        this.resourceRoutines = resourceRoutines;
    }

    @ZAction(defaultAction = true)
    @Path("/displays")
    public AssignedRoomList index() {
        return findRooms();
    }

    @Path("/displays/unassign")
    public void unassign(@RequestParam("id") long resourceId) {
        resourceRoutines.updateResourceDisplayTokenById(resourceId, null);
        throw new ZRedirectException("/displays");
    }

    @Path("/displays/autodecline")
    public void autodecline(@RequestParam("id") long resourceId, @RequestParam("off") boolean off) {
        resourceRoutines.updateResourceAutoDeclineFromDisplay(resourceId, !off);
        throw new ZRedirectException("/displays");
    }

    private AssignedRoomList findRooms() {
        Comparator<Resource> resourceC = Resource.getF(ResourceFields.OFFICE_ID).andThenNaturalComparator()
                .thenComparing(Resource.getF(ResourceFields.FLOOR_NUM).andThenNaturalComparator().nullLowC())
                .thenComparing(Resource.getF(ResourceFields.EXCHANGE_NAME).andThenNaturalComparator().nullLowC())
                .uncheckedCastC();

        ListF<ResourceInfo> resources = resourceRoutines.getYtRoomsHaveDisplayToken()
                .sorted(ResourceInfo.resourceF().andThen(resourceC));

        return new AssignedRoomList(resources.map(Room.fromResourceInfoF(
                resourceRoutines.getAutoDeclineFromDisplayResourceIds())));
    }

    @BenderBindAllFields
    public static final class AssignedRoomList {
        @BenderPart(name = "room")
        public final ListF<Room> rooms;

        public AssignedRoomList(ListF<Room> rooms) {
            this.rooms = rooms;
        }
    }

    @BenderBindAllFields
    public static final class Room {
        public final long id;
        public final String name;
        public final String officeName;
        public final Option<Long> officeStaffId;

        public final Option<Instant> lastPing;
        public final boolean broken;
        public final boolean active;
        public final boolean autoDecline;

        public Room(long id, String name, String officeName, Option<Long> officeStaffId, Option<Instant> lastPing, boolean active, boolean autoDecline) {
            this.id = id;
            this.name = name;
            this.officeName = officeName;
            this.officeStaffId = officeStaffId;
            this.lastPing = lastPing;
            this.broken = !lastPing.isPresent() || lastPing.get().isBefore(Instant.now().minus(Duration.standardMinutes(5)));
            this.active = active;
            this.autoDecline = autoDecline;
        }

        public static Function<ResourceInfo, Room> fromResourceInfoF(ListF<Long> autoDeclineIds) {
            return r -> new Room(
                    r.getResourceId(),
                    r.getNameWithAlterName().getOrElse(""),
                    r.getOffice().getName(),
                    r.getOfficeStaffId(),
                    r.getResource().getDisplayLastPingTs(),
                    r.getResource().getIsActive(),
                    autoDeclineIds.containsTs(r.getResourceId()));
        }
    }
}
