package ru.yandex.calendar.admin.exchange;

import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemTypeType;
import com.microsoft.schemas.exchange.services._2006.types.ExtendedPropertyType;
import com.microsoft.schemas.exchange.services._2006.types.UnindexedFieldURIType;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.admin.exchange.ExchangeEventsLookup.ExchangeEventsLookupByEmails;
import ru.yandex.calendar.admin.exchange.ExchangeEventsLookup.LookupByExchangeEmail;
import ru.yandex.calendar.admin.exchange.ExchangeEventsLookup.LookupByExchangeId;
import ru.yandex.calendar.admin.exchange.ExchangeEventsLookup.LookupInSubscribedResources;
import ru.yandex.calendar.frontend.ews.EwsSubjectNotSubscribedException;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.ExchangeEmailManager;
import ru.yandex.calendar.frontend.ews.YtEwsSubscriptionDao;
import ru.yandex.calendar.frontend.ews.compare.EwsComparator;
import ru.yandex.calendar.frontend.ews.compare.EwsCompareResult;
import ru.yandex.calendar.frontend.ews.compare.EwsCompareResultBrief;
import ru.yandex.calendar.frontend.ews.exp.EwsExportRoutines;
import ru.yandex.calendar.frontend.ews.imp.ExchangeEventDataConverter;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxyWrapper;
import ru.yandex.calendar.frontend.ews.sync.IgnoredEventDao;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.YtEwsIgnoredEvent;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.repetition.RegularRepetitionRule;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

public class ExchangeAdminManager {
    private static final Logger logger = LoggerFactory.getLogger(ExchangeAdminPage.class);

    @Autowired
    private EwsProxyWrapper ewsProxyWrapper;
    @Autowired
    private IgnoredEventDao ignoredEventDao;
    @Autowired
    private ExchangeEmailManager exchangeEmailManager;
    @Autowired
    private EwsComparator ewsComparator;
    @Autowired
    private YtEwsSubscriptionDao ytEwsSubscriptionDao;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private EwsExportRoutines ewsExportRoutines;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private ResourceRoutines resourceRoutines;

    public static class ItemInfo {
        public Option<YtEwsIgnoredEvent> ignoredEventO;
        public ListF<ExtendedPropertyType> extProperties;
        public Option<CalendarItemType> item;
    }

    public ItemInfo getItemInfo(String id, boolean isParentByOccurrenceId) {
        ItemInfo res = new ItemInfo();

        // ssytnik: not using res.item's id because event can be deleted in exchange
        res.ignoredEventO = ignoredEventDao.findIgnoredEventByExchangeId(id);
        res.item = isParentByOccurrenceId ?
            ewsProxyWrapper.getMasterOrSingleEvent(id) :
            ewsProxyWrapper.getEvent(id);
        res.extProperties = res.item.isPresent() ?
            Cf.x(res.item.get().getExtendedProperty()) :
            Cf.list();

        return res;
    }


    public ListF<CalendarItemType> getMastersOrInstancesInInterval(
            InstantInterval interval, Email exchangeEmail, boolean isInstancesLookup)
    {
        ListF<UnindexedFieldURIType> fields = Cf.list(
                UnindexedFieldURIType.CALENDAR_UID,
                UnindexedFieldURIType.ITEM_SUBJECT,
                UnindexedFieldURIType.CALENDAR_START,
                UnindexedFieldURIType.CALENDAR_END,
                UnindexedFieldURIType.CALENDAR_CALENDAR_ITEM_TYPE,
                UnindexedFieldURIType.CALENDAR_ORGANIZER,
                UnindexedFieldURIType.CALENDAR_REQUIRED_ATTENDEES,
                UnindexedFieldURIType.CALENDAR_OPTIONAL_ATTENDEES,
                UnindexedFieldURIType.CALENDAR_RESOURCES,
                UnindexedFieldURIType.ITEM_LAST_MODIFIED_TIME,
                UnindexedFieldURIType.CALENDAR_IS_CANCELLED);

        ListF<String> exchangeIds = isInstancesLookup ?
            ewsProxyWrapper.findInstanceEventIds(exchangeEmail, interval) :
            ewsProxyWrapper.findMasterAndSingleEventIds(exchangeEmail, Option.of(interval));

        return ewsProxyWrapper.getEvents(exchangeIds, fields);
    }

    public boolean isSubscribed(Email email) {
        try {
            exchangeEmailManager.getExchangeEmailByEmail(email);
            return true;
        } catch (EwsSubjectNotSubscribedException e) {
            return false;
        }
    }

    public ListF<EwsCompareResult> getCompareResults(Option<Email> emailO, Option<Integer> daysO) {
        InstantInterval interval = daysO.isPresent() ?
                EwsComparator.getIntervalDays(daysO.get()) : EwsComparator.getDefaultInterval();

        return emailO.isPresent() ?
            Cf.list(ewsComparator.compare(emailO.get(), interval)) :
            ewsComparator.compareRandomSubjects(1, interval);
    }

    public EwsCompareResultBrief getEwsLoginsCompareResultBrief(Option<Integer> daysO) {
        InstantInterval interval = daysO.isPresent() ?
                EwsComparator.getIntervalDays(daysO.get()) : EwsComparator.getDefaultInterval();
        return ewsComparator.compareEwsLogins(interval);
    }

    /** (name, startO) => count of such meetings in mailbox */
    public static class CopiesInfo {
        public Tuple2List<Tuple2<String, Option<Instant>>, Integer> copies;
    }

    public CopiesInfo getTopCopiesInfo(Email exchangeEmail) {
        CopiesInfo res = new CopiesInfo();
        res.copies = Cf2.stableGroupBy(ewsProxyWrapper.findMasterAndSingleEvents(exchangeEmail, Option.empty()),
            a -> Tuple2.tuple(a.getSubject(), EwsUtils.toInstantO(a.getStart()))).entries().map2(Cf.List.sizeF()).takeSortedBy2Desc(100);
        return res;
    }

    public ListF<Email> getExchangeEmails(ExchangeEventsLookupByEmails lookup) {
        Check.C.isTrue(lookup instanceof LookupInSubscribedResources || lookup instanceof LookupByExchangeEmail);
        ListF<Email> exchangeEmails;
        if (lookup instanceof LookupInSubscribedResources) {
            exchangeEmails = ytEwsSubscriptionDao
                    .findSubscribedResources()
                    .map(exchangeEmailManager.getExchangeEmailBySubscriptionF());
            LookupInSubscribedResources lookupInSubscribedResources = (LookupInSubscribedResources) lookup;
            if (lookupInSubscribedResources.getOffsetO().isPresent()) {
                exchangeEmails = exchangeEmails.drop(lookupInSubscribedResources.getOffsetO().get());
            }
            if (lookupInSubscribedResources.getLimitO().isPresent()) {
                exchangeEmails = exchangeEmails.take(lookupInSubscribedResources.getLimitO().get());
            }
        } else {
            exchangeEmails = Cf.list(((LookupByExchangeEmail) lookup).getExchangeEmail());
        }
        return exchangeEmails;
    }

    public ListF<String> getExchangeIds(
            ExchangeEventsLookup lookup,
            Function<ExchangeEventsLookupByEmails, ExchangeEventInfos> f)
    {
        if (lookup instanceof LookupByExchangeId) {
            return Cf.list( ((LookupByExchangeId) lookup).getExchangeId() );
        } else {
            Check.C.isTrue(lookup instanceof ExchangeEventsLookupByEmails);
            ExchangeEventInfos infos = f.apply((ExchangeEventsLookupByEmails) lookup);
            ListF<ExchangeEventInfo> flatInfos = infos.infosByExchangeEmail.get2().flatten();
            return flatInfos.map(ExchangeEventInfo.getExchangeIdF());
        }
    }


    public static class ExchangeEventInfo {
        public String name;
        public Instant start;
        public Option<Tuple2<RegularRepetitionRule, Option<Instant>>> repetitionTypeAndDueO;
        public String exchangeId;
        public Email mailbox;

        @Override
        public String toString() {
            return
                "(" + name + ", " + start + ", " + repetitionTypeAndDueO + ", " +
                exchangeId + ", " + mailbox + ")";
        }

        public static Function<ExchangeEventInfo, String> getExchangeIdF() {
            return a -> a.exchangeId;
        }
    };

    public static class ExchangeEventInfos {
        public Tuple2List<Email, ListF<ExchangeEventInfo>> infosByExchangeEmail;
    }


    public Function<ExchangeEventsLookupByEmails, ExchangeEventInfos> getUtcMastersNoDuesF() {
        return a -> getUtcMasters(a, false);
    }

    public ExchangeEventInfos getUtcMasters(ExchangeEventsLookupByEmails lookup, boolean withDues) {
        return getEventsByLookupAndFilter(lookup, withDues,
                a -> isUtcItemWithType(a, CalendarItemTypeType.RECURRING_MASTER));
    }

    public static boolean isUtcItemWithType(CalendarItemType item, CalendarItemTypeType type) {
        final ListF<String> utcAliases = Cf.list("UTC", "GMT", "GMT Standard Time", "Greenwich Standard Time");
        return item.getCalendarItemType() == type && utcAliases.containsTs(item.getTimeZone());
    }

    public Function<ExchangeEventsLookupByEmails, ExchangeEventInfos> getAllDaysNoDuesF() {
        return a -> getAllDays(a, false);
    }

    public ExchangeEventInfos getAllDays(ExchangeEventsLookupByEmails lookup, boolean withDues) {
        return getEventsByLookupAndFilter(lookup, withDues, ExchangeAdminManager::isAllDay);
    }

    public static boolean isAllDay(CalendarItemType item) {
        return item.isIsAllDayEvent() != null && item.isIsAllDayEvent();
    }

    private ExchangeEventInfos getEventsByLookupAndFilter(
            final ExchangeEventsLookupByEmails lookup, final boolean withDues,
            final Function1B<CalendarItemType> f)
    {
        ListF<Email> exchangeEmails = getExchangeEmails(lookup);

        ExchangeEventInfos res = new ExchangeEventInfos();
        res.infosByExchangeEmail = Cf.Tuple2List.arrayList();
        for (final Email exchangeEmail : exchangeEmails) {
            logger.info("Current email = " + exchangeEmail);

            ListF<ExchangeEventInfo> infos = ewsProxyWrapper
                    .findMasterAndSingleEvents(exchangeEmail, Option.empty())
                    .filter(f)
                    .map(a -> {
                        ExchangeEventInfo res1 = new ExchangeEventInfo();
                        res1.name = a.getSubject();
                        res1.start = EwsUtils.xmlGregorianCalendarInstantToInstant(a.getStart());
                        res1.exchangeId = a.getItemId().getId();
                        res1.mailbox = exchangeEmail;

                        res1.repetitionTypeAndDueO = Option.empty();
                        if (withDues) {
                            Option<CalendarItemType> fullO = ewsProxyWrapper.getEvent(res1.exchangeId);
                            if (fullO.isPresent()) {
                                CalendarItemType full = fullO.get();
                                EventData eventData = ExchangeEventDataConverter
                                        .convertWithoutExchangeDataAndRealOrganizer(
                                                full, resourceRoutines::selectResourceEmails);
                                Repetition repetition = eventData.getRepetition();
                                if (repetition == null || RepetitionRoutines.isNoneRepetition(repetition)) {
                                    res1.repetitionTypeAndDueO = Option.empty();
                                } else {
                                    res1.repetitionTypeAndDueO = Option.of(Tuple2.tuple(
                                        repetition.getType(),
                                        repetition.getDueTs()
                                    ));
                                }
                            } else {
                                logger.warn("Event not found for id: " + res1.exchangeId);
                            }
                        }
                        return res1;
                    }).sortedBy((Function<ExchangeEventInfo, Instant>) a -> a.start);

            res.infosByExchangeEmail.add(exchangeEmail, infos);

        } // for

        return res;
    }


    public static class FixUtcMastersInfo {
        public String exchangeId;
        public long eventId0;
        public Event masterEvent;
        public DateTimeZone creatorTz;
        public Tuple2List<ResourceParticipantInfo, Option<CalendarItemType>> stateBefore;
        public Tuple2List<ResourceParticipantInfo, Option<CalendarItemType>> stateAfter;
    }

    public ListF<Either<FixUtcMastersInfo, Exception>> fixUtcMasters(ExchangeEventsLookup lookup) {
        ListF<String> exchangeIds = getExchangeIds(lookup, getUtcMastersNoDuesF());

        return exchangeIds.map(exchangeId -> {
            try {
                FixUtcMastersInfo res = new FixUtcMastersInfo();
                res.exchangeId = exchangeId;
                res.eventId0 = eventRoutines.findEventByExchangeId(exchangeId)
                        .getOrThrow("Could not find event by exchange id: " + exchangeId).getId();
                res.masterEvent = eventDao.findMasterEventByEventId(res.eventId0).single();
                res.creatorTz = dateTimeManager.getTimeZoneForUid(res.masterEvent.getCreatorUid());

                ListF<ResourceParticipantInfo> eventResources = resourceDao
                        .findYaTeamResourceLayersThatSyncWithExchangeWithEvents(Cf.list(res.masterEvent.getId()));
                res.stateBefore = zipResourcesWithEvents(eventResources);

                ewsExportRoutines.changeTimezoneInExchangeExistingMasterEvents(
                        res.masterEvent.getId(), res.creatorTz,
                        new ActionInfo(ActionSource.UNKNOWN, "?", Instant.now()));

                res.stateAfter = zipResourcesWithEvents(eventResources);
                return Either.left(res);
            } catch (Exception e) {
                return Either.right(e);
            }
        });
    }

    public Tuple2List<ResourceParticipantInfo, Option<CalendarItemType>> zipResourcesWithEvents(
            ListF<ResourceParticipantInfo> resources)
    {
        return resources.zipWith(a -> {
            if (a.hasExchangeId()) {
                return ewsProxyWrapper.getEvent(a.getExchangeId().get());
            } else {
                return Option.empty();
            }
        });
    }


    public static class FixAllDaysInfo {
        public String exchangeId;
        public long eventId0;
    }

    public ListF<Either<FixAllDaysInfo, Exception>> fixAllDays(ExchangeEventsLookup lookup) {
        ListF<String> exchangeIds = getExchangeIds(lookup, getAllDaysNoDuesF());

        return exchangeIds.map(exchangeId -> {
            try {
                FixAllDaysInfo res = new FixAllDaysInfo();
                res.exchangeId = exchangeId;
                res.eventId0 = eventRoutines.findEventByExchangeId(exchangeId)
                        .getOrThrow("Could not find event by exchange id: " + exchangeId).getId();

                // TODO ssytnik@, gutman@: set (not adjust) proper start/end to a calendar event

                return Either.left(res);
            } catch (Exception e) {
                return Either.right(e);
            }
        });
    }
}
