package ru.yandex.calendar.admin.exchange;

import java.util.concurrent.TimeUnit;

import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import org.dom4j.Element;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.admin.exchange.ExchangeAdminManager.CopiesInfo;
import ru.yandex.calendar.admin.exchange.ExchangeAdminManager.ExchangeEventInfo;
import ru.yandex.calendar.admin.exchange.ExchangeAdminManager.ExchangeEventInfos;
import ru.yandex.calendar.admin.exchange.ExchangeAdminManager.FixAllDaysInfo;
import ru.yandex.calendar.admin.exchange.ExchangeAdminManager.FixUtcMastersInfo;
import ru.yandex.calendar.admin.exchange.ExchangeAdminManager.ItemInfo;
import ru.yandex.calendar.admin.exchange.ExchangeEventsLookup.ExchangeEventsLookupByEmails;
import ru.yandex.calendar.boot.EwsAliveHandler;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.ExchangeEmailManager;
import ru.yandex.calendar.frontend.ews.compare.EwsCompareResult;
import ru.yandex.calendar.frontend.ews.imp.ExchangeEventDataConverter;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.util.serialize.ToMultilineSerializer;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.MoscowTime;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

@ActionContainer
public class ExchangeAdminPage {

    private final ExchangeAdminManager exchangeAdminManager;
    private final ExchangeEmailManager exchangeEmailManager;
    private final ToMultilineSerializer toMultilineSerializer;
    private final EwsAliveHandler ewsAliveHandler;

    public ExchangeAdminPage(ExchangeAdminManager exchangeAdminManager, ExchangeEmailManager exchangeEmailManager,
            ToMultilineSerializer toMultilineSerializer, EwsAliveHandler ewsAliveHandler)
    {
        this.exchangeAdminManager = exchangeAdminManager;
        this.exchangeEmailManager = exchangeEmailManager;
        this.toMultilineSerializer = toMultilineSerializer;
        this.ewsAliveHandler = ewsAliveHandler;
    }

    @ZAction(defaultAction = true)
    @Path("/exchange")
    public Element index(@SpecialParam HttpServletRequestX req) {
        return index(req, Option.empty());
    }

    @ZAction(timeout = 2, timeoutUnit = TimeUnit.MINUTES)
    @Path("/exchange/{action}")
    public Element index(
            @SpecialParam
            HttpServletRequestX req,
            @PathParam("action")
            Option<String> action)
    {
        Option<String> output = getOutput(req, action.getOrElse(""));
        Element r = Dom4jUtils.createElement("output");
        r.addText(output.getOrElse("<output is empty>") + "\n$");
        return r;
    }

    private Option<String> getOutput(HttpServletRequestX req, String action) {
        if (!ewsAliveHandler.isEwsAlive()) return Option.empty();
        try {
            if ("item".equals(action)) {
                String id = req.getParameter("id");
                boolean isParentByOccurrenceId = "1".equals(req.getParameter("parent-by-occurrence-id"));
                boolean isDetailedView = "1".equals(req.getParameter("detailed-view"));

                return outItemInfo(exchangeAdminManager.getItemInfo(id, isParentByOccurrenceId), isDetailedView);

            } else if ("interval".equals(action)) {
                LocalDate startDate = TimeUtils.localDate.parse(req.getParameter("start-date"));
                LocalDate endDate = TimeUtils.localDate.parseSafe(req.getParameter("end-date")).getOrElse(startDate);

                InstantInterval interval = new InstantInterval( // ssytnik: can use email's timezone here
                        startDate.toDateTimeAtStartOfDay(MoscowTime.TZ).toInstant(),
                        endDate.toDateTimeAtStartOfDay(MoscowTime.TZ).plusDays(1).toInstant());

                Email email = getEmailParameter(req, "email");
                Email exchangeEmail = exchangeEmailManager.getExchangeEmailByAnyEmailOrGetAsIs(email);

                String itemType = req.getParameterO("item-type").getOrElse("instances");
                boolean isInstancesLookup = "instances".equals(itemType);

                ListF<CalendarItemType> mastersOrInstances = exchangeAdminManager.getMastersOrInstancesInInterval(
                        interval, exchangeEmail, isInstancesLookup);

                return Option.of(outMastersOrInstances(interval, exchangeEmail, itemType, mastersOrInstances));

            } else if ("misc".equals(action) && req.getParameterO("compare-random").isPresent()) {

                return Option.of(outCompareResults(
                        exchangeAdminManager.getCompareResults(Option.<Email>empty(), Option.<Integer>empty())));

            } else if ("misc".equals(action) && req.getParameterO("check-subscribed").isPresent()) {
                Email email = getEmailParameter(req, "email");

                return Option.of(outIsSubscribed(exchangeAdminManager.isSubscribed(email)));

            } else if ("misc".equals(action)
                    && (req.getParameterO("compare").isPresent() || req.getParameterO("compare-ews").isPresent()))
            {
                Option<Integer> daysO =
                        StringUtils.notEmptyO(req.getParameter("misc-compare-days")).map(Cf.Integer::parse);

                if (req.getParameterO("compare-ews").isPresent()) {
                    return Option.of(String.valueOf(exchangeAdminManager.getEwsLoginsCompareResultBrief(daysO)));
                } else {
                    Email email = getEmailParameter(req, "email");
                    return Option.of(outCompareResults(
                            exchangeAdminManager.getCompareResults(Option.of(email), daysO)));
                }

            } else if ("misc".equals(action) && req.getParameterO("top-copies").isPresent()) {
                Email email = getEmailParameter(req, "email");
                Email exchangeEmail = exchangeEmailManager.getExchangeEmailByAnyEmailOrGetAsIs(email);

                CopiesInfo topCopiesInfo = exchangeAdminManager.getTopCopiesInfo(exchangeEmail);

                return Option.of(outEventCopiesInfo(exchangeEmail, topCopiesInfo));

            } else if ("misc".equals(action) && req.getParameterO("filter-utc-masters").isPresent()) {
                ExchangeEventsLookupByEmails lookup = getExchangeEventsLookupByEmails(req);
                boolean withDues = "1".equals(req.getParameter("misc-with-dues"));

                ExchangeEventInfos infos = exchangeAdminManager.getUtcMasters(lookup, withDues);
                return Option.of(infos.infosByExchangeEmail.map(outExchangeEventInfosF()).mkString("\n==\n\n"));

            } else if ("misc".equals(action) && req.getParameterO("filter-all-days").isPresent()) {
                ExchangeEventsLookupByEmails lookup = getExchangeEventsLookupByEmails(req);
                boolean withDues = "1".equals(req.getParameter("misc-with-dues"));

                ExchangeEventInfos infos = exchangeAdminManager.getAllDays(lookup, withDues);
                return Option.of(infos.infosByExchangeEmail.map(outExchangeEventInfosF()).mkString("\n==\n\n"));

            } else if ("misc".equals(action) && req.getParameterO("fix-utc-masters").isPresent()) {
                ExchangeEventsLookup lookup = getExchangeEventsLookup(req, true);

                ListF<Either<FixUtcMastersInfo, Exception>> infosOrEx = exchangeAdminManager.fixUtcMasters(lookup);
                return Option.of(infosOrEx.map(outFixUtcMastersInfoOrExF()).mkString("\n==\n\n"));

            } else if ("misc".equals(action) && req.getParameterO("fix-all-days").isPresent()) {
                ExchangeEventsLookup lookup = getExchangeEventsLookup(req, true);

                ListF<Either<FixAllDaysInfo, Exception>> infosOrEx = exchangeAdminManager.fixAllDays(lookup);
                return Option.of(infosOrEx.map(outFixAllDaysInfoOrExF()).mkString("\n==\n\n"));

            } else {

                return Option.empty();
            }

        } catch (Exception e) {
            return Option.of("ERROR: " + ExceptionUtils.getStackTrace(e));
        }
    }


    private Email getEmailParameter(HttpServletRequestX req, String name) {
        return getEmailParameterO(req, name).getOrThrow("Could not get email from parameter: " + name);
    }

    private Option<Email> getEmailParameterO(HttpServletRequestX req, String name) {
        return StringUtils.notEmptyO(req.getParameter(name)).map(Email.consF());
    }

    private ExchangeEventsLookupByEmails getExchangeEventsLookupByEmails(HttpServletRequestX req) {
        return (ExchangeEventsLookupByEmails) getExchangeEventsLookup(req, false);
    }

    private ExchangeEventsLookup getExchangeEventsLookup(HttpServletRequestX req, boolean allowExchangeId) {
        Option<Email> emailO = getEmailParameterO(req, "email");
        boolean isEmail = emailO.isPresent();

        boolean isResources = "1".equals(req.getParameter("misc-all-resources"));

        Option<String> exchangeIdO = req.getParameterO("misc-exchange-id").filter(Cf.String.notEmptyF());
        boolean isExchangeId = allowExchangeId && exchangeIdO.isPresent();

        int lookupSourcesCount = (int) (Binary.toLong(isEmail) + Binary.toLong(isResources) + Binary.toLong(isExchangeId));
        switch (lookupSourcesCount) {
            case 0: {
                throw new IllegalArgumentException("please specify lookup source");
            }
            case 1: {
                break;
            }
            default: {
                throw new IllegalArgumentException("ambiguity: please specify single lookup source");
            }
        }

        if (isEmail) {
            Email exchangeEmail = exchangeEmailManager.getExchangeEmailByAnyEmailOrGetAsIs(emailO.get());
            return ExchangeEventsLookup.lookupByExchangeEmail(exchangeEmail);
        } else if (isResources) {
            Option<Integer> offsetO = StringUtils.notEmptyO(req.getParameter("misc-offset")).map(Cf.Integer::parse);
            Option<Integer> limitO = StringUtils.notEmptyO(req.getParameter("misc-limit")).map(Cf.Integer::parse);
            return ExchangeEventsLookup.lookupInSubscribedResources(offsetO, limitO);
        } else if (isExchangeId) {
            return ExchangeEventsLookup.lookupByExchangeId(exchangeIdO.get());
        } else {
            throw new IllegalStateException();
        }
    }


    private Option<String> outItemInfo(ItemInfo info, boolean isDetailedView) {
        Option<String> output = Option.empty();

        if (info.ignoredEventO.isPresent()) {
            output = Option.of(output.getOrElse("") + "NOTE: found " + info.ignoredEventO.get() + "\n\n");
        }

        if (info.extProperties.isNotEmpty()) {
            output = Option.of(
                    output.getOrElse("") + "NOTE: found extended properties: " +
                            toMultilineSerializer.serialize(info.extProperties) + "\n\n");
        }

        if (info.item.isPresent()) {
            output = Option.of(output.getOrElse("") + itemToStringFull(info.item.get(), isDetailedView));
        }

        return output;
    }

    private String itemToStringFull(CalendarItemType calItem, boolean isDetailedView) {
        String res = toMultilineSerializer.serialize(calItem);

        if (!isDetailedView) {
            final ListF<String> skippedHeaders = Cf.list(
                    "adjacentMeetings",
                    "conflictingMeetings",
                    "internetMessageHeaders",
                    "responseObjects"
            ).map(Cf.String.plusF().bind2(" ="));

            String[] lines = res.split("\n");
            ListF<String> newLines = Cf.arrayList();
            boolean isAdding = true;
            for (String line : lines) {
                if (line.charAt(0) != ' ') {
                    isAdding = !skippedHeaders.containsTs(line);
                }
                if (isAdding) {
                    newLines.add(line);
                }
            }
            res = newLines.mkString("\n");
        }

        return res;
    }


    private String outMastersOrInstances(
            InstantInterval interval, Email exchangeEmail,
            String itemType, ListF<CalendarItemType> mastersOrInstances)
    {
        return itemType + " lookup for " + exchangeEmail + " @ " + interval + ":\n\n" +
                itemsToString(mastersOrInstances);
    }

    // XXX copy-paste from EwsImporter + changes
    private String itemsToString(ListF<CalendarItemType> calItems) {
        StringBuilder sb = new StringBuilder();
        for (CalendarItemType calItem : calItems) {
            sb.append(LogMarker.EXCHANGE_ID.format(calItem.getItemId().getId()));
            sb.append('\n');
            sb.append(LogMarker.EXT_ID.format(StringUtils.defaultIfEmpty(calItem.getUID(), "?")));
            sb.append('\n');
            sb.append("Item type: " + (calItem.isIsCancelled() ? "CANCELED, " : "") + calItem.getCalendarItemType());
            sb.append('\n');
            sb.append("Event name: " + calItem.getSubject());
            sb.append('\n');
            sb.append("Event start timestamp: " + EwsUtils.xmlGregorianCalendarInstantToInstant(calItem.getStart()));
            sb.append('\n');
            try {
                final Option<Email> organizerEmail = ExchangeEventDataConverter.getOrganizerEmailSafe(calItem);
                sb.append("Organizer: " + organizerEmail);
            } catch (Exception e) {
                sb.append("Could not get organizer email: " + e.toString());
            }
            sb.append('\n');
            try {
                final ListF<Email> attendeeEmails = ExchangeEventDataConverter.getAttendeeEmails(calItem);
                sb.append("Attendees: " + toMultilineSerializer.serialize(attendeeEmails));
            } catch (Exception e) {
                sb.append("Could not get attendee emails: " + e.toString());
            }
            sb.append('\n');
            sb.append("Event last-modified: " + EwsUtils.xmlGregorianCalendarInstantToInstant(calItem.getLastModifiedTime()));
            sb.append("\n\n");
        }
        return sb.toString();
    }

    private String outCompareResults(ListF<EwsCompareResult> compareResults) {
        return toMultilineSerializer.serialize(compareResults);
    }

    private String outIsSubscribed(boolean isSubscribed) {
        return isSubscribed ?
                "Subject is subscribed, you can use any form of this email for queries (either @ld, @y-t or @msft)" :
                "Subject is not subscribed, you should know exact subject's exchange email to get proper results";
    }

    private String outEventCopiesInfo(Email exchangeEmail, CopiesInfo copiesInfo) {
        return "top copies lookup for " + exchangeEmail + ":\n\n" + copiesInfo.copies.mkString("\n");
    }

    private Function2<Email, ListF<ExchangeEventInfo>, String> outExchangeEventInfosF() {
        return this::outExchangeEventInfo;
    }

    private String outExchangeEventInfo(Email email, ListF<ExchangeEventInfo> exchangeEventInfos) {
        return "repeating UTC events lookup for " + email + ":\n\n" + exchangeEventInfos.mkString("\n");
    }

    private Function<Either<FixUtcMastersInfo, Exception>, String> outFixUtcMastersInfoOrExF() {
        return this::outFixUtcMastersInfoOrEx;
    }

    private String outFixUtcMastersInfoOrEx(Either<FixUtcMastersInfo, Exception> infoOrEx) {
        if (infoOrEx.isLeft()) {
            FixUtcMastersInfo info = infoOrEx.getLeft();
            StringBuilder sb = new StringBuilder();
            sb.append("looked up event id by exchangeId = " + info.exchangeId + "\n");
            sb.append("looked up master event by eventId = " + info.eventId0 + "\n");
            sb.append("found master event: " + info.masterEvent +
                    "\nid = " + info.masterEvent.getId() + ", " +
                    "creator_uid = " + info.masterEvent.getCreatorUid() + ", " +
                    "name = " + info.masterEvent.getName() +
                    "\napplying new timezone: " + info.creatorTz + "\n");
            appendResourcesWithTimezones(sb, "before update", info.stateBefore);
            sb.append("\n");
            appendResourcesWithTimezones(sb, "after update", info.stateAfter);
            return sb.toString();
        } else {
            return "ERROR: " + infoOrEx.getRight();
        }
    }

    private void appendResourcesWithTimezones(
            StringBuilder sb, String header, Tuple2List<ResourceParticipantInfo, Option<CalendarItemType>> state)
    {
        sb.append("\n" + header + ":");
        for (Tuple2<ResourceParticipantInfo, Option<CalendarItemType>> t : state) {
            sb.append("\n");
            if (t.get1().hasExchangeId()) {
                if (t.get2().isPresent()) {
                    sb.append("tz = " + t.get2().get().getTimeZone());
                } else {
                    sb.append("event not found by exchangeId (" + t.get1().getExchangeId().get() + ")");
                }
            } else {
                sb.append("event exchangeId unknown/not exists");
            }
            sb.append(" - ");
            sb.append(t.get1());
        }
    }

    private Function<Either<FixAllDaysInfo, Exception>, String> outFixAllDaysInfoOrExF() {
        return this::outFixAllDaysInfoOrEx;
    }

    private String outFixAllDaysInfoOrEx(Either<FixAllDaysInfo, Exception> infoOrEx) {
        if (infoOrEx.isLeft()) {
            FixAllDaysInfo info = infoOrEx.getLeft();
            StringBuilder sb = new StringBuilder();
            sb.append("looked up event id by exchangeId = " + info.exchangeId + "\n");
            sb.append("looked up master event by eventId = " + info.eventId0 + "\n");
            sb.append("TODO see ExchangeAdminManager.fixAllDays()");
            return sb.toString();
        } else {
            return "ERROR: " + infoOrEx.getRight();
        }
    }
}
