package ru.yandex.calendar.admin.resources;

import org.dom4j.Element;

import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.commune.util.universal.UniversalObjectParser;
import ru.yandex.inside.passport.blackbox.PassportDomain;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.enums.StringEnumResolver;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * @author ssytnik
 * TODO idempotency for *_do operations (via TL-cache with random keys)
 */
@ActionContainer
public class ResourcesAdminPage {
    private static final Logger logger = LoggerFactory.getLogger(ResourcesAdminPage.class);
    private static final boolean DRY_RUN = false;

    private final ResourcesManager resourcesManager;
    private final UniversalObjectParser universalObjectParser;

    public ResourcesAdminPage(ResourcesManager resourcesManager, UniversalObjectParser universalObjectParser) {
        this.resourcesManager = resourcesManager;
        this.universalObjectParser = universalObjectParser;
    }

    @ZAction(defaultAction = true)
    @Path("/rooms")
    public Element index(@SpecialParam HttpServletRequestX req) {
        return index(req, Option.<Mode>empty());
    }

    @ZAction
    @Path("/rooms/{mode}")
    public Element index(
            @SpecialParam
            HttpServletRequestX req,
            @PathParam("mode")
            Option<Mode> modeO)
    {
        Mode mode = modeO.getOrElse(Mode.VIEW);

        switch (mode) {
            case CREATE_DO: {
                Resource resource = parseResource(req, true);
                if (DRY_RUN) {
                    logger.warn("Dry-run: not creating " + resource);
                } else {
                    // XXX 'number' is set manually according to building plan
                    // XXX 'pos' should be <floor>*100 + <number>
                    // XXX 'exchange_name' should be conf_rr_<floor>_<number>
                    resourcesManager.createResource(resource);
                }
                mode = Mode.VIEW;
                break;
            }
            case EDIT_DO: {
                Resource resource = parseResource(req, false);
                if (DRY_RUN) {
                    logger.warn("Dry-run: not updating " + resource);
                } else {
                    resourcesManager.updateResource(resource);
                }
                mode = Mode.VIEW;
                break;
            }
            case DELETE_DO: {
                long id = Long.parseLong(req.getParameter("id"));
                if (DRY_RUN) {
                    logger.warn("Dry-run: not deleting " + id);
                } else {
                    resourcesManager.deleteResource(id);
                }
                mode = Mode.VIEW;
                break;
            }
        }

        Element rMode = Dom4jUtils.createElement(mode.value());

        switch (mode) {
            case VIEW: {
                Element resourcesListElement = Dom4jUtils.createElement("resources-list");
                for (Resource resource : resourcesManager.getResources()) {
                    resourcesListElement.add(xmlizeForView(resource));
                }
                rMode.add(resourcesListElement);
                break;
            }
            case CREATE: {
                rMode.add(xmlizeForCreate());
                break;
            }
            case EDIT: {
                long id = Long.parseLong(req.getParameter("id"));
                Resource resource = resourcesManager.getResourceById(id);
                rMode.add(xmlizeForEdit(resource));
                break;
            }
        }

        return rMode;
    }


    private static enum Mode implements StringEnum {
        VIEW,
        CREATE,
        CREATE_DO,
        EDIT,
        EDIT_DO,
        DELETE_DO,
        ;

        public static final StringEnumResolver<Mode> R = StringEnumResolver.r(Mode.class);

        @Override
        public String value() {
            return CamelWords.parse(this.name()).toDbName();
        }
    }

    private static String getFieldXmlName(MapField<?> field) {
        return CamelWords.parse(field.getName()).toXmlName();
    }

    private static void addFieldElement(Element resourceElement, MapField<?> field, Object value) {
        resourceElement.addElement(getFieldXmlName(field))
                .addAttribute("required", String.valueOf(!field.isPartOfId() && field.isNotNull()))
                .addAttribute("type", String.valueOf(field.getType().getSimpleName()))
                .setText(ObjectUtils.toString(value));
    }


    private static Element xmlizeForView(Resource resource) {
        Element resourceElement = Dom4jUtils.createElement("resource");
        resourceElement.addAttribute("id", String.valueOf(resource.getId()));
        resourceElement.addAttribute("name", resource.getName().getOrElse(""));
        resourceElement.addAttribute("exchange-name", resource.getExchangeName().getOrElse(""));
        resourceElement.addAttribute("is-active", String.valueOf(resource.getIsActive()));
        return resourceElement;
    }

    private static Element xmlizeForCreate() {
        MapF<MapField<?>, Object> defaults = Tuple2List.<MapField<?>, Object>fromPairs(
                ResourceFields.DOMAIN, PassportDomain.YANDEX_TEAM_RU,
                ResourceFields.IS_ACTIVE, Boolean.TRUE
        ).toMap();
        Element resourceElement = Dom4jUtils.createElement("resource");
        for (MapField<?> field : ResourceFields.OBJECT_DESCRIPTION.getFields()) {
            Object defaultValue = field.isNotNull() && field.getType().isPrimitiveWrapper() ?
                    field.getType().unwrapPrimitiveWrapper().defaultValue() : null;
            Object value = defaults.getO(field).getOrElse(defaultValue);
            addFieldElement(resourceElement, field, value);
        }
        return resourceElement;
    }

    private static Element xmlizeForEdit(Resource resource) {
        Element resourceElement = Dom4jUtils.createElement("resource");
        for (MapField<?> field : ResourceFields.OBJECT_DESCRIPTION.getFields()) {
            Object value = resource.getFieldValue(field);
            addFieldElement(resourceElement, field, value);
        }
        return resourceElement;
    }


    private Resource parseResource(HttpServletRequestX req, boolean forCreate) {
        Resource r = new Resource();
        for (MapField<?> field : ResourceFields.OBJECT_DESCRIPTION.getFields()) {
            if (forCreate && field.isPartOfId()) {
                continue;
            }
            String valueStr = req.getParameter(getFieldXmlName(field));
            Object value = StringUtils.isEmpty(valueStr) && !field.isNotNull() ?
                    null : universalObjectParser.parse(valueStr, field.getType());
            r.setFieldValue(field.<Object>cast(), value);
        }
        return r;
    }
}
