package ru.yandex.calendar.admin.specialusers;

import java.util.EnumSet;
import java.util.stream.Stream;

import lombok.val;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.IntegerArray;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.beans.generated.UserGroups;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.user.Group;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserGroupsDao;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.ParameterDescriptor;
import ru.yandex.commune.a3.action.parameter.WebRequest;
import ru.yandex.commune.a3.action.parameter.bind.ParameterBinder;
import ru.yandex.commune.a3.action.parameter.bind.annotation.BindWith;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.admin.z.ZRedirectException;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.annotation.BenderTextValue;
import ru.yandex.misc.bender.annotation.XmlAttribute;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.test.Assert;

import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

@ActionContainer
public class SpecialUsersAdminPage {

    private final UserGroupsDao userGroupsDao;
    private final UserManager userManager;
    private final SettingsRoutines settingsRoutines;
    private final ResourceDao resourceDao;
    private final Option<String> upravlyatorUrl;

    public SpecialUsersAdminPage(
            UserGroupsDao userGroupsDao,
            UserManager userManager, SettingsRoutines settingsRoutines,
            ResourceDao resourceDao, Option<String> upravlyatorUrl) {
        this.userGroupsDao = userGroupsDao;
        this.userManager = userManager;
        this.settingsRoutines = settingsRoutines;
        this.resourceDao = resourceDao;
        this.upravlyatorUrl = upravlyatorUrl;
    }

    @ZAction(defaultAction = true)
    @Path("/special-users")
    public UserInfoList index() {
        ListF<UserGroups> groups = userGroupsDao.findAll();
        ListF<SpecialUserInfo> users = getUserInfos(groups.filter(g -> g.getUid().isPresent()))
                .sortedBy(SpecialUserInfo.getEmailF());

        ListF<DepartmentInfo> departments = getDepartmentInfos(groups.filter(g -> g.getDepartmentUrl().isPresent()))
                .sortedBy(DepartmentInfo.getDepartmentF());

        return new UserInfoList(users, departments, getRoles(), upravlyatorUrl);
    }

    @ZAction(engineId = ZAction.TEXT)
    @Path("/special-users/create")
    public String create(
            @RequestParam("emailOrDepartment")
                    String emailOrDepartment,
            @BindWith(TurnedOnGroupsBinder.class)
                    ListF<GroupOrRole> turnedOnGroups) {
        return save(emailOrDepartment, Cf.list(), turnedOnGroups);
    }

    @ZAction(engineId = ZAction.TEXT)
    @Path("/special-users/save")
    public String save(
            @RequestParam("emailOrDepartment")
                    String emailOrDepartment,
            @BindWith(TurnedOnGroupsBinder.class)
                    ListF<GroupOrRole> turnedOnGroups) {
        var knownGroups = getRoles().map(r -> r.name);
        return this.save(emailOrDepartment, knownGroups, turnedOnGroups);
    }

    private String save(String emailOrDepartment, ListF<String> knownGroups, ListF<GroupOrRole> turnedOnGroups) {
        Assert.none(upravlyatorUrl, "Read only");
        ListF<GroupOrRole> known = knownGroups.map(GroupOrRole::parse);
        ListF<GroupOrRole> turnedOff = known.isEmpty() ? Cf.list() : known.filter(turnedOnGroups.containsF().notF());

        if (Emails.isEmail(emailOrDepartment)) {
            PassportUid uid = userManager.getUidByEmail(new Email(emailOrDepartment)).getOrThrow("user not found");
            saveOrUpdateUser(uid, turnedOnGroups, turnedOff);
        } else {
            saveOrUpdateDepartment(emailOrDepartment, turnedOnGroups, turnedOff);
        }
        throw new ZRedirectException("/special-users");
    }

    private ListF<RoleInfo> getRoles() {
        ListF<RoleInfo> roles = Cf.list(
                new RoleInfo("super-user", "Super user", Option.of("label-success")),
                new RoleInfo("meeting-room-admin", "Room admin", Option.of("label-success")),
                new RoleInfo("massage-admin", "Massage admin", Option.of("label-danger")),
                new RoleInfo("parking-admin", "Parking admin", Option.of("label-info")),
                new RoleInfo("apartment-user", "Apartment user", Option.of("label-warning"), Option.of("apartment")),
                new RoleInfo("apartment-admin", "Apartment admin", Option.of("label-warning"), Option.of("apartment")),
                new RoleInfo("hotel-user", "Hotel user", Option.of("label-blue"), Option.of("hotel")),
                new RoleInfo("hotel-admin", "Hotel admin", Option.of("label-blue"), Option.of("hotel")),
                new RoleInfo("campus-admin", "Campus admin", Option.of("label-blue")));

        return roles.plus(resourceDao.findResources(ResourceFields.ACCESS_GROUP.column().isNotNull())
                .groupBy(r -> r.getAccessGroup().get()).values()
                .flatMap(rs -> {
                    ListF<String> names = rs.map(r -> r.getNameEn().orElse(r.getName()).get());
                    String name = (names.size() < 3 ? names : names.take(2).plus1("…")).mkString(", ");

                    String group = Integer.toString(rs.first().getAccessGroup().get());

                    return Cf.list(
                            new RoleInfo(group + "-user", name + " user", Option.empty(), Option.of(group)),
                            new RoleInfo(group + "-admin", name + " admin", Option.empty(), Option.of(group)));
                }));
    }

    private void saveOrUpdateDepartment(String departmentUrl, ListF<GroupOrRole> turnedOn, ListF<GroupOrRole> turnedOff) {
        val currentDepartmentGroups = userGroupsDao.findByDepartmentUrls(Cf.list(departmentUrl)).singleO();

        val newDepartmentGroups = new UserGroups();
        currentDepartmentGroups.map(UserGroups::getId).forEach(newDepartmentGroups::setId);
        newDepartmentGroups.setDepartmentUrl(departmentUrl);
        newDepartmentGroups.setGroups(groups(currentDepartmentGroups, turnedOn, turnedOff));
        newDepartmentGroups.setResourcesCanAccess(resourceCanAccess(currentDepartmentGroups, turnedOn, turnedOff));
        newDepartmentGroups.setResourcesCanAdmin(resourceCanAdmin(currentDepartmentGroups, turnedOn, turnedOff));
        userGroupsDao.saveOrUpdate(newDepartmentGroups);
    }

    private void saveOrUpdateUser(PassportUid uid, ListF<GroupOrRole> turnedOn, ListF<GroupOrRole> turnedOff) {
        val currentUserGroups = userGroupsDao.findByUids(Cf.list(uid)).singleO();

        val newUserGroups = new UserGroups();
        currentUserGroups.map(UserGroups::getId).forEach(newUserGroups::setId);
        newUserGroups.setUid(uid);
        newUserGroups.setGroups(groups(currentUserGroups, turnedOn, turnedOff));
        newUserGroups.setResourcesCanAccess(resourceCanAccess(currentUserGroups, turnedOn, turnedOff));
        newUserGroups.setResourcesCanAdmin(resourceCanAdmin(currentUserGroups, turnedOn, turnedOff));
        userGroupsDao.saveOrUpdate(newUserGroups);
    }

    private IntegerArray groups(
            Option<UserGroups> existingGroups, ListF<GroupOrRole> turnedOn, ListF<GroupOrRole> turnedOff) {
        val turnedOffGroups = turnedOff.stream()
                .flatMap(gr -> gr.getGroupO().stream())
                .collect(toSet());

        val existingGroupsStream = existingGroups.stream()
                .map(UserGroups.getGroupsF())
                .map(Group::integerArrayToEnumSet)
                .flatMap(EnumSet::stream);

        val turnedOnGroupsStream = turnedOn.stream().flatMap(gr -> gr.getGroupO().stream());

        val resultGroups = Stream.concat(existingGroupsStream, turnedOnGroupsStream)
                .filter(v -> !turnedOffGroups.contains(v))
                .map(Group::value)
                .distinct()
                .collect(CollectorsF.toList());

        return new IntegerArray(resultGroups);
    }

    private IntegerArray resourceCanAccess(
            Option<UserGroups> existingGroups, ListF<GroupOrRole> turnedOn, ListF<GroupOrRole> turnedOff) {
        return resourceGroups(existingGroups, turnedOn, turnedOff,
                UserGroups::getResourcesCanAccess, Role::getIdUser);
    }

    private IntegerArray resourceCanAdmin(
            Option<UserGroups> existingGroups, ListF<GroupOrRole> turnedOn, ListF<GroupOrRole> turnedOff) {
        return resourceGroups(existingGroups, turnedOn, turnedOff,
                UserGroups::getResourcesCanAdmin, Role::getIdAdmin);
    }

    private IntegerArray resourceGroups(
            Option<UserGroups> existingGroups, ListF<GroupOrRole> turnedOn, ListF<GroupOrRole> turnedOff,
            Function<UserGroups, IntegerArray> extractResources, Function<Role, Option<Integer>> getRoleId) {
        val offGroupsNumbers = turnedOff.stream()
                .flatMap(gr -> gr.getRoleO().stream())
                .flatMap(r -> getRoleId.apply(r).stream())
                .collect(toSet());

        val existingGroupsStream = existingGroups.stream()
                .map(extractResources)
                .map(IntegerArray::getElements)
                .flatMap(ListF::stream);

        val turnedOnGroupsStream = turnedOn.stream()
                .flatMap(gr -> gr.getRoleO().stream())
                .flatMap(r -> getRoleId.apply(r).stream());

        val existingAndTurnOnGroupsNumbers = Stream.concat(existingGroupsStream, turnedOnGroupsStream)
                .distinct()
                .filter(v -> !offGroupsNumbers.contains(v))
                .collect(CollectorsF.toList());
        return new IntegerArray(existingAndTurnOnGroupsNumbers);
    }

    private static ListF<Role> getRoles(UserGroups groups) {
        return Stream.concat(groups.getResourcesCanAccess().getElements().stream().map(resource -> new Role(Either.left(resource))),
                groups.getResourcesCanAdmin().getElements().stream().map(resource -> new Role(Either.right(resource)))
        ).collect(CollectorsF.toList());
    }

    private ListF<SpecialUserInfo> getUserInfos(ListF<UserGroups> userGroups) {
        val uids = userGroups.stream()
                .flatMap(g -> g.getUid().stream())
                .collect(CollectorsF.toList());
        val emailByUid = settingsRoutines.getSettingsCommonByUidBatch(uids)
                .mapValues(Settings.getEmailF());

        val ytLoginByUid = userGroups.stream()
                .flatMap(g -> g.getUid().stream())
                .filter(PassportUid::isYandexTeamRu)
                .collect(toMap(v -> v, userManager::getYtUserLoginByUid));

        return userGroups.map(groups -> {
            PassportUid uid = groups.getUid().get();
            return new SpecialUserInfo(
                    emailByUid.getOrThrow(uid).getEmail(), uid, ytLoginByUid.get(uid),
                    Cf.x(Group.integerArrayToEnumSet(groups.getGroups())).toList(), getRoles(groups));
        });
    }

    private ListF<DepartmentInfo> getDepartmentInfos(ListF<UserGroups> departmentGroups) {
        return departmentGroups.map(g -> new DepartmentInfo(
                g.getDepartmentUrl().get(),
                Cf.toList(Group.integerArrayToEnumSet(g.getGroups())), getRoles(g)));
    }

    @Bendable
    public static final class UserInfoList {
        @BenderPart(name = "user-info")
        public final ListF<SpecialUserInfo> users;
        @BenderPart(name = "department-info")
        public final ListF<DepartmentInfo> departments;
        @BenderPart(name = "role-info")
        public final ListF<RoleInfo> roles;
        @BenderPart
        public final Option<String> upravlyatorUrl;

        public UserInfoList(
                ListF<SpecialUserInfo> users,
                ListF<DepartmentInfo> departments,
                ListF<RoleInfo> roles, Option<String> upravlyatorUrl) {
            this.users = users;
            this.departments = departments;
            this.roles = roles;
            this.upravlyatorUrl = upravlyatorUrl;
        }
    }

    @Bendable
    public static class SpecialUserInfo {
        @XmlAttribute
        public final PassportUid uid;
        @XmlAttribute
        public final String email;
        @XmlAttribute
        public final Option<String> ytLogin;
        @BenderPart(name = "group", wrapperName = "groups")
        public final ListF<Group> groups;
        @BenderPart(name = "role", wrapperName = "roles")
        public final ListF<Role> roles;

        private SpecialUserInfo(
                String email, PassportUid uid, Option<String> ytLogin, ListF<Group> groups, ListF<Role> roles) {
            this.uid = uid;
            this.email = email;
            this.ytLogin = ytLogin;
            this.groups = groups;
            this.roles = roles;
        }

        public static Function<SpecialUserInfo, String> getEmailF() {
            return u -> u.email;
        }
    }

    @Bendable
    public static class DepartmentInfo {
        @XmlAttribute
        public final String department;
        @BenderPart(name = "group", wrapperName = "groups")
        public final ListF<Group> groups;
        @BenderPart(name = "role", wrapperName = "roles")
        public final ListF<Role> roles;

        public DepartmentInfo(String department, ListF<Group> groups, ListF<Role> roles) {
            this.department = department;
            this.groups = groups;
            this.roles = roles;
        }

        public static Function<DepartmentInfo, String> getDepartmentF() {
            return d -> d.department;
        }
    }

    @Bendable
    public static class RoleInfo {
        @XmlAttribute
        private final String name;
        @XmlAttribute
        private final String displayName;
        @XmlAttribute
        private final String labelClass;
        @XmlAttribute
        private final Option<String> group;

        public RoleInfo(String name, String displayName, Option<String> labelClass) {
            this(name, displayName, labelClass, Option.empty());
        }

        public RoleInfo(String name, String displayName, Option<String> labelClass, Option<String> group) {
            this.name = name;
            this.displayName = displayName;
            this.labelClass = labelClass.getOrElse("label-default");
            this.group = group;
        }
    }

    public static class GroupOrRole extends DefaultObject {
        private final Either<Group, Role> either;

        public GroupOrRole(Either<Group, Role> either) {
            this.either = either;
        }

        public Option<Group> getGroupO() {
            return either.leftO();
        }

        public Option<Role> getRoleO() {
            return either.rightO();
        }

        public static GroupOrRole parse(String value) {
            return new GroupOrRole(Either.fromOptions(Group.R.valueOfO(value), Role.parseSafe(value)));
        }

        public static Option<GroupOrRole> parseSafe(String value) {
            try {
                return Option.of(parse(value));
            } catch (IllegalArgumentException e) {
                return Option.empty();
            }
        }
    }

    @BenderBindAllFields
    public static class Role extends DefaultObject {
        private Either<Integer, Integer> userOrAdminId;

        public Role(Either<Integer, Integer> userOrAdminId) {
            this.userOrAdminId = userOrAdminId;
        }

        public Option<Integer> getIdUser() {
            return userOrAdminId.leftO();
        }

        public Option<Integer> getIdAdmin() {
            return userOrAdminId.rightO();
        }

        @BenderTextValue
        public String serialize() {
            return userOrAdminId.fold(i -> i + "-user", i -> i + "-admin");
        }

        public static Option<Role> parseSafe(String value) {
            Option<Integer> user = Cf.Integer.parseSafe(StringUtils.substringBefore(value, "-user"));
            Option<Integer> admin = Cf.Integer.parseSafe(StringUtils.substringBefore(value, "-admin"));

            return Option.when(user.plus(admin).isNotEmpty(), () -> new Role(Either.fromOptions(user, admin)));
        }
    }

    public static class TurnedOnGroupsBinder implements ParameterBinder {
        public Object createAndBind(
                WebRequest webRequest,
                ActionInvocationContext invocationContext,
                ParameterDescriptor parameterDescriptor) {
            return webRequest.getHttpServletRequest().getParameterMapF().entrySet()
                    .filterMap(e -> GroupOrRole.parseSafe(e.getKey()).filter(p -> e.getValue().containsTs("true")));
        }
    }
}
