package ru.yandex.calendar.admin.todos;

import org.dom4j.Element;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.stat.TodoDailyStat;
import ru.yandex.calendar.logic.stat.TodoStatisticsManager;
import ru.yandex.calendar.logic.stat.YandexMailTodoDailyStat;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.admin.z.ZRedirectException;
import ru.yandex.commune.json.jackson.ObjectMapperX;
import ru.yandex.commune.web.action.ErrorXmlizer;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * @author Daniel Brylev
 */
@ActionContainer
public class TodoStatsAdminPage {

    private final TodoStatisticsManager todoStatisticsManager;
    private final ObjectMapperX objectMapper;

    public TodoStatsAdminPage(TodoStatisticsManager todoStatisticsManager, ObjectMapperX objectMapper) {
        this.todoStatisticsManager = todoStatisticsManager;
        this.objectMapper = objectMapper;
    }

    @ZAction(defaultAction = true)
    @Path("/todo-stats")
    public Element index() {
        return index("kpi");
    }

    @ZAction
    @Path("/todo-stats/{tab}")
    public Element index(
            @PathParam("tab")
            String tab)
    {
        Element content = Dom4jUtils.createElement("content");
        ListF<TodoDailyStat> stats = todoStatisticsManager.getStats().sortedBy(TodoDailyStat::getDate);

        if (tab.equalsIgnoreCase("kpi")) {
            content.add(activeUsersChartElement(stats));
            content.add(activeUsersIncrementChartElement(stats));

            try {
                MapF<LocalDate, YandexMailTodoDailyStat> mailStatByDate = todoStatisticsManager.getYandexMailStats()
                        .toMapMappingToKey(YandexMailTodoDailyStat.getDateF());
                Tuple2List<TodoDailyStat, YandexMailTodoDailyStat> joined =
                        stats.zipWithFlatMapO(s -> mailStatByDate.getO(s.getDate()));
                content.add(serviceUsageChartElement(joined));

            } catch (Exception e) {
                content.add(chartErrorElement(e));
            }
        } else if (tab.equalsIgnoreCase("audience")) {
            content.add(usersCreatedTodoItemsChartElement(stats));
            content.add(usersCreatedTodoListsChartElement(stats));

        } else if (tab.equalsIgnoreCase("detailed")) {
            content.add(createdTodoItemsWithDueTsChartElement(stats));
            content.add(usersCreatedTodoItemsWithDueTsChartElement(stats));
            content.add(usersSentTodoEmailsChartElement(stats));

        } else {
            throw new ZRedirectException("/todo-stats");
        }

        ListF<Element> tabElements = Cf.list(
                Dom4jUtils.createElement("tab").addAttribute("id", "kpi").addAttribute("title", "KPI"),
                Dom4jUtils.createElement("tab").addAttribute("id", "audience").addAttribute("title", "Аудитория"),
                Dom4jUtils.createElement("tab").addAttribute("id", "detailed").addAttribute("title", "Подробно"));

        Element tabsElement = Dom4jUtils.createElement("tabs");

        for (Element tabElement : tabElements) {
            if (tabElement.attributeValue("id").equalsIgnoreCase(tab)) {
                tabElement.addAttribute("active", "1");
            }
            tabsElement.add(tabElement);
        }
        content.add(tabsElement);

        return content;
    }

    private Element usersCreatedTodoItemsChartElement(ListF<TodoDailyStat> stats) {
        ListF<?> data = stats.map(new Function<TodoDailyStat, Object>() {
            public Object apply(final TodoDailyStat stat) {
                return new Object() {
                    public final String date = stat.getDate().toString();
                    public final int from1To20 = stat.getUsersCreated1To20TodoItemsByWeek();
                    public final int from21To34 = stat.getUsersCreated21To34TodoItemsByWeek();
                    public final int from35To55 = stat.getUsersCreated35To55TodoItemsByWeek();
                    public final int from56To76 = stat.getUsersCreated56To76TodoItemsByWeek();
                    public final int from76 = stat.getUsersCreatedMoreThan76TodoItemsByWeek();
                };
            }
        });
        String title = "Количество пользователей, создававших дела (в среднем за неделю)";

        ListF<Element> graphs = Cf.list(
                graphElement("до 2 дел в день", "from1To20"),
                graphElement("от 3 до 4 дел в день", "from21To34"),
                graphElement("от 5 до 7 дел в день", "from35To55"),
                graphElement("от 8 до 10 дел в день", "from56To76"),
                graphElement("более 10 дел в день", "from76")
        );
        return chartElement(title, "date", data, graphs);
    }

    private Element createdTodoItemsWithDueTsChartElement(ListF<TodoDailyStat> stats) {
        ListF<?> data = stats.map(new Function<TodoDailyStat, Object>() {
            public Object apply(final TodoDailyStat stat) {
                return new Object() {
                    public final String date = stat.getDate().toString();
                    public final int withDueTs = stat.getCreatedTodoItemsWithDueTsByDay();
                    public final int total = stat.getCreatedTodoItemsByDay();
                    public final float ratio = total > 0 ? 10000 * withDueTs / total / 100f : 100f;
                };
            }
        });
        String title = "Количество созданных дел (за день)";

        ListF<Element> graphs = Cf.list(
                graphElement("всего дел", "total"),
                graphElement("дел с датой окончания", "withDueTs"),
                graphElement("процентное отношение дел с датой окончания к общему количеству", "ratio")
        );
        return chartElement(title, "date", data, graphs);
    }

    private Element usersCreatedTodoItemsWithDueTsChartElement(ListF<TodoDailyStat> stats) {
        ListF<?> data = stats.map(new Function<TodoDailyStat, Object>() {
            public Object apply(final TodoDailyStat stat) {
                return new Object() {
                    public final String date = stat.getDate().toString();
                    public final int withDueTs = stat.getUsersCreatedTodoItemWithDueTsByDay();
                    public final int total = stat.getUsersCreatedTodoItemByDay();
                    public final float ratio = total > 0 ? 10000 * withDueTs / total / 100f : 100f;
                };
            }
        });
        String title = "Количество пользователей, создавших хотя бы одно дело (за день)";

        ListF<Element> graphs = Cf.list(
                graphElement("создавших любое дело", "total"),
                graphElement("создавших дело с датой окончания", "withDueTs"),
                graphElement("процентное отношение создавших дело с датой окончания к создавшим любое дело", "ratio")
        );
        return chartElement(title, "date", data, graphs);
    }

    private Element usersSentTodoEmailsChartElement(ListF<TodoDailyStat> stats) {
        ListF<?> data = stats.map(new Function<TodoDailyStat, Object>() {
            public Object apply(final TodoDailyStat stat) {
                return new Object() {
                    public final String date = stat.getDate().toString();
                    public final int value = stat.getUsersSentTodoListEmailEver();
                };
            }
        });
        String title = "Количество пользователей, отправивших список дел письмом (за всё время)";

        return chartElement(title, "date", data, graphElement("количество пользователей", "value"));
    }

    private Element usersCreatedTodoListsChartElement(ListF<TodoDailyStat> stats) {
        ListF<?> data = stats.map(new Function<TodoDailyStat, Object>() {
            public Object apply(final TodoDailyStat stat) {
                return new Object() {
                    public final String date = stat.getDate().toString();
                    public final int oneOrMore = stat.getUsersCreatedTodoListEver();
                    public final int twoOrMore = stat.getUsersCreatedMoreThanOneTodoListEver();
                };
            }
        });
        String title = "Количество пользователей, создавших списки дел (за всё время)";

        ListF<Element> graphs = Cf.list(
                graphElement("хотя бы один список", "oneOrMore"),
                graphElement("два или более списка", "twoOrMore")
        );
        return chartElement(title, "date", data, graphs);
    }

    private Element activeUsersChartElement(ListF<TodoDailyStat> stats) {
        ListF<?> data = stats.map(new Function<TodoDailyStat, Object>() {
            public Object apply(final TodoDailyStat stat) {
                return new Object() {
                    public final String date = stat.getDate().toString();
                    public final int inTwoWeeks = stat.getUsersCreatedOrModifiedTodoItemBy2Weeks();
                    public final int inMonth = stat.getUsersCreatedOrModifiedTodoItemByMonth();
                };
            }
        });
        String title = "Количество пользователей, выполнивших активное действие" +
                " (создание, редактирование или удаление дела)";

        ListF<Element> graphs = Cf.list(
                graphElement("за последние 14 дней", "inTwoWeeks"),
                graphElement("за последние 30 дней", "inMonth")
        );
        return chartElement(title, "date", data, graphs);
    }

    private Element activeUsersIncrementChartElement(ListF<TodoDailyStat> stats) {
        ListF<?> data = stats.zip(stats.drop(1)).map(new Function2<TodoDailyStat, TodoDailyStat, Object>() {
            public Object apply(final TodoDailyStat yesterdayStat, final TodoDailyStat todayStat) {
                return new Object() {
                    public final String date = todayStat.getDate().toString();
                    public final int inTwoWeeks = todayStat.getUsersCreatedOrModifiedTodoItemBy2Weeks()
                            - yesterdayStat.getUsersCreatedOrModifiedTodoItemBy2Weeks();
                    public final int inMonth = todayStat.getUsersCreatedOrModifiedTodoItemByMonth()
                            - yesterdayStat.getUsersCreatedOrModifiedTodoItemByMonth();
                };
            }
        });
        String title = "Ежедневный прирост количества пользователей, выполнивших активное действие";

        ListF<Element> graphs = Cf.list(
                graphElement("за последние 14 дней", "inTwoWeeks"),
                graphElement("за последние 30 дней", "inMonth")
        );
        return chartElement(title, "date", data, graphs);
    }

    private Element serviceUsageChartElement(Tuple2List<TodoDailyStat, YandexMailTodoDailyStat> stats) {
        ListF<?> data = stats.map(new Function2<TodoDailyStat, YandexMailTodoDailyStat, Object>() {
            public Object apply(final TodoDailyStat stat, final YandexMailTodoDailyStat mailStat) {
                return new Object() {
                    public final String date = stat.getDate().toString();
                    private final int d = mailStat.getUsersWithShowTodoSettingOn();

                    public final float inTwoWeeks = stat.getUsersCreatedOrModifiedTodoItemBy2Weeks() * 10000 / d / 100f;
                    public final float inMonth = stat.getUsersCreatedOrModifiedTodoItemByMonth() * 10000 / d / 100f;
                };
            }
        });
        String title = "Процентное отношение к количеству пользователей с включенной настройкой ToDo" +
                "\\nколичества пользователей, выполнивших активное действие";

        ListF<Element> graphs = Cf.list(
                graphElement("за последние 14 дней", "inTwoWeeks"),
                graphElement("за последние 30 дней", "inMonth")
        );
        return chartElement(title, "date", data, graphs);
    }

    private Element graphElement(String title, String valueField) {
        Element graph = Dom4jUtils.createElement("graph");
        graph.addAttribute("title", title);
        graph.addAttribute("value-field", valueField);
        return graph;
    }

    private Element chartElement(String title, String categoryField, ListF<?> data, Element graphElement) {
        return chartElement(title, categoryField, data, Cf.list(graphElement));
    }

    private Element chartElement(String title, String categoryField, ListF<?> data, ListF<Element> graphElements) {
        Element chart = Dom4jUtils.createElement("chart");
        chart.addAttribute("title", title);
        chart.addAttribute("category-field", categoryField);
        chart.add(Dom4jUtils.createElementWithContent("data", objectMapper.writeValueAsString(data)));
        chart.content().addAll(graphElements);
        return chart;
    }

    private Element chartErrorElement(Throwable throwable) {
        Element chart = Dom4jUtils.createElement("chart");
        chart.add(ErrorXmlizer.xmlize(throwable));
        return chart;
    }

}
