package ru.yandex.calendar.boot;

import javax.annotation.PostConstruct;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.val;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.calendar.CalendarDataSourceStatus;
import ru.yandex.calendar.CalendarVersion;
import ru.yandex.calendar.admin.universal.CalendarUniversalContextConfiguration;
import ru.yandex.calendar.frontend.display.DisplayManagerContextConfiguration;
import ru.yandex.calendar.frontend.ews.ExchangeClientContextConfiguration;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxyChecker;
import ru.yandex.calendar.frontend.ews.sync.OrganizerCacheManager;
import ru.yandex.calendar.frontend.worker.task.UpdateLayersTimestampsTask;
import ru.yandex.calendar.frontend.xiva.XivaContextConfiguration;
import ru.yandex.calendar.logic.LastUpdateManager;
import ru.yandex.calendar.logic.contact.ContactContextConfiguration;
import ru.yandex.calendar.logic.domain.DomainManager;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.EventContextConfiguration;
import ru.yandex.calendar.logic.event.avail.AvailabilityContextConfiguration;
import ru.yandex.calendar.logic.ics.IcsContextConfiguration;
import ru.yandex.calendar.logic.layer.LayerContextConfiguration;
import ru.yandex.calendar.logic.notification.NotificationContextConfiguration;
import ru.yandex.calendar.logic.notification.NotificationRoutines;
import ru.yandex.calendar.logic.organization.OrganizationManager;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceAccessRoutines;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.SpecialResources;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleContextConfiguration;
import ru.yandex.calendar.logic.sending.SendingContextConfiguration;
import ru.yandex.calendar.logic.sending.param.EventMessageInfoCreator;
import ru.yandex.calendar.logic.sending.real.PassportSmsServiceMock;
import ru.yandex.calendar.logic.sharding.ShardingLocator;
import ru.yandex.calendar.logic.sharing.perm.PermContextConfiguration;
import ru.yandex.calendar.logic.suggest.SuggestContextConfiguration;
import ru.yandex.calendar.logic.svc.DbSvcRoutines;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.telemost.TelemostContextConfiguration;
import ru.yandex.calendar.logic.todo.TodoContextConfiguration;
import ru.yandex.calendar.logic.update.UpdateContextConfiguration;
import ru.yandex.calendar.logic.user.BlackboxNgContextConfiguration;
import ru.yandex.calendar.logic.user.CenterContextConfiguration;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.SpecialUsers;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.logic.user.UserRoutines;
import ru.yandex.calendar.monitoring.DynamicMonitoringContextConfiguration;
import ru.yandex.calendar.monitoring.EwsSensors;
import ru.yandex.calendar.unistat.UnistatConfiguration;
import ru.yandex.calendar.util.HttpClientWithMetrics;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.validation.Captcha;
import ru.yandex.calendar.zk.CalendarZkContextConfiguration;
import ru.yandex.commune.dynproperties.DynamicPropertiesContextConfiguration;
import ru.yandex.commune.dynproperties.DynamicPropertyBenderMapperHolder;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.commune.util.serialize.ToMultilineSerializerContextConfiguration;
import ru.yandex.inside.admin.conductor.ConductorContextConfiguration;
import ru.yandex.inside.captcha.CaptchaService;
import ru.yandex.inside.passport.YandexPassport;
import ru.yandex.inside.passport.sms.PassportSmsService;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.parse.simpleType.LocalDateUnmarshaller;
import ru.yandex.misc.bender.serialize.simpleType.LocalDateMarshaller;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.version.Version;

import static ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils.multiThreadedClient;

@Configuration
@Import({
        MicroCoreContextConfiguration.class,
        CalendarJdbcContextConfiguration.class,
        CalendarBazingaClientContextConfiguration.class,
        CalendarZkContextConfiguration.class,
        ToMultilineSerializerContextConfiguration.class,
        CenterContextConfiguration.class, // XXX: will fail for yandex.ru
        IcsContextConfiguration.class,
        SendingContextConfiguration.class,
        EventContextConfiguration.class,
        AvailabilityContextConfiguration.class,
        LayerContextConfiguration.class,
        TodoContextConfiguration.class,
        NotificationContextConfiguration.class,
        ContactContextConfiguration.class,
        UpdateContextConfiguration.class,
        PermContextConfiguration.class,
        ExchangeClientContextConfiguration.class,
        BlackboxNgContextConfiguration.class,
        CalendarUniversalContextConfiguration.class,
        ResourceScheduleContextConfiguration.class,
        ConductorContextConfiguration.class,
        DynamicPropertiesContextConfiguration.class,
        SuggestContextConfiguration.class,
        XivaContextConfiguration.class,
        DisplayManagerContextConfiguration.class,
        UnistatConfiguration.class,
        DynamicMonitoringContextConfiguration.class,
        TelemostContextConfiguration.class,
})
public class CalendarContextConfiguration {

    @PostConstruct
    public void init() {
        AuxDateTime.checkJdkDefaultTimeZoneIsUtc();
    }

    @Bean
    public Version version() {
        return CalendarVersion.VERSION;
    }

    @Bean
    public DomainManager domainManager() {
        return new DomainManager();
    }

    @Bean
    public PassportSmsService passportSmsService(MeterRegistry registry,
            @Value("${sms.passport.timeout}") int passportSmsTimeout,
            @Value("${sms.passport.max_connections}") int passportSmsMaxConnections,
            EnvironmentType environmentType) {
        if (environmentType == EnvironmentType.TESTS) {
            return new PassportSmsServiceMock();
        }

        val httpClient = new HttpClientWithMetrics(
                multiThreadedClient(Timeout.seconds(passportSmsTimeout), passportSmsMaxConnections), registry, "sms");
        return new PassportSmsService(httpClient);
    }

    @Bean
    public YandexPassport yandexPassport() {
        return new YandexPassport();
    }

    @Bean
    public CaptchaService captchaService() {
        return new CaptchaService();
    }

    @Bean
    public Captcha captcha() {
        return new Captcha();
    }

    @Bean
    public PassportAuthDomainsHolder passportAuthDomainsHolder() {
        return PassportAuthDomainsHolder.H;
    }

    @Bean
    public OfficeManager officeManager() {
        return new OfficeManager();
    }

    @Bean
    public OrganizationManager organizationManager() {
        return new OrganizationManager();
    }

    @Bean
    public UserManager userManager(DynamicPropertyManager dynamicPropertyManager) {
        dynamicPropertyManager.addStaticFields(SpecialUsers.class);

        return new UserManager();
    }

    @Bean
    public ShardingLocator shardingLocator() {
        return new ShardingLocator();
    }

    @Bean
    public DateTimeManager dateTimeManager() {
        return new DateTimeManager();
    }

    @Bean
    public DbSvcRoutines dbSvcRoutines() {
        return new DbSvcRoutines();
    }

    @Bean
    public NotificationRoutines notificationRoutines() {
        return new NotificationRoutines();
    }

    @Bean
    public DynamicPropertyBenderMapperHolder benderMapperHolder() {
        return new DynamicPropertyBenderMapperHolder(new BenderMapper(BenderConfiguration.cons(
                MembersToBind.WITH_ANNOTATIONS, false,
                CustomMarshallerUnmarshallerFactoryBuilder.cons()
                        .add(LocalDate.class, new LocalDateMarshaller(), new LocalDateUnmarshaller())
                        .build())));
    }

    @Bean
    public ResourceRoutines resourceRoutines(DynamicPropertyManager dynamicPropertyManager) {
        dynamicPropertyManager.addStaticFields(SpecialResources.class);

        return new ResourceRoutines();
    }

    @Bean
    public SettingsRoutines settingsRoutines() {
        return new SettingsRoutines();
    }

    @Bean
    public UserRoutines userRoutines() {
        return new UserRoutines();
    }

    @Bean
    public SvcRoutines svcRoutines() {
        return new SvcRoutines();
    }

    @Bean
    public OrganizerCacheManager organizerCacheManager() {
        return new OrganizerCacheManager();
    }

    @Bean
    public EventMessageInfoCreator eventMessageInfoCreator() {
        return new EventMessageInfoCreator();
    }

    @Bean
    public LastUpdateManager lastUpdateManager() {
        return new LastUpdateManager();
    }

    @Bean
    public UpdateLayersTimestampsTask updateLayersTimestampsTask() {
        return new UpdateLayersTimestampsTask();
    }

    @Bean
    public CalendarDataSourceStatus dataSourceStatus() {
        return new CalendarDataSourceStatus();
    }

    @Bean
    public ResourceAccessRoutines eventResourceRoutines() {
        return new ResourceAccessRoutines();
    }

    @Bean
    public EwsSensors ewsSensors(MeterRegistry registry, EwsProxyChecker ewsProxyChecker) {
        return new EwsSensors(registry, ewsProxyChecker);
    }

    @Bean
    public EwsAliveHandler ewsAliveHandler(DynamicPropertyManager dynamicPropertyManager) {
        dynamicPropertyManager.addStaticFields(EwsAliveHandler.class);
        return new EwsAliveHandler();
    }
}
