package ru.yandex.calendar.boot;

import java.sql.Connection;

import javax.sql.DataSource;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.val;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Either;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.unistat.UnistatConfiguration;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.calendar.util.db.CalendarJdbcTemplate;
import ru.yandex.calendar.util.db.CalendarQueryInterceptors;
import ru.yandex.calendar.util.db.DataSourceFactoryWithMetrics;
import ru.yandex.calendar.util.db.DcAwareDynamicMasterSlaveDataSourceFactoryBean;
import ru.yandex.calendar.util.db.ListenableTransactionTemplate;
import ru.yandex.calendar.util.db.ReadFromSlaveQueryInterceptor;
import ru.yandex.calendar.util.db.TransactionListener;
import ru.yandex.calendar.util.db.TransactionLogger;
import ru.yandex.commune.alive2.location.LocationResolverConfiguration;
import ru.yandex.inside.admin.conductor.Conductor;
import ru.yandex.inside.admin.conductor.ConductorContextConfiguration;
import ru.yandex.misc.db.pool.dbcp.DbcpConfiguration;
import ru.yandex.misc.db.postgres.PgBouncerFamiliarTransactionManager;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.spring.ApplicationContextUtils;

@Configuration
@ComponentScan(basePackages = "ru.yandex.calendar", useDefaultFilters = false, includeFilters =
@ComponentScan.Filter(type = FilterType.ASSIGNABLE_TYPE, value = CalendarJdbcDaoSupport.class))
@Import({
        UnistatConfiguration.class,
        ConductorContextConfiguration.class,
        LocationResolverConfiguration.class
})
public class CalendarJdbcContextConfiguration {

    @Bean
    public TransactionTemplate transactionTemplate(
            PlatformTransactionManager transactionManager, ApplicationContext context) {
        DefaultTransactionDefinition definition = new DefaultTransactionDefinition();
        definition.setIsolationLevel(Connection.TRANSACTION_READ_COMMITTED);

        return new ListenableTransactionTemplate(
                transactionManager, definition,
                ApplicationContextUtils.beansOfType(context, TransactionListener.class));
    }

    @Bean
    public TransactionListener eventsTskvLoggerTransactionListener() {
        return EventsLogger.createTransactionListener();
    }

    @Bean
    public TransactionLogger transactionLogger() {
        return new TransactionLogger();
    }

    @Bean
    public ReadFromSlaveQueryInterceptor readSlaveInterceptor(CalendarAppName appName) {
        return new ReadFromSlaveQueryInterceptor(appName.xmlName());
    }

    @Bean
    public CalendarJdbcTemplate jdbcTemplate(DataSource ds, ReadFromSlaveQueryInterceptor readSlaveInterceptor) {
        return new CalendarJdbcTemplate(ds, CalendarQueryInterceptors.create(ds, readSlaveInterceptor));
    }

    @Bean
    public PlatformTransactionManager transactionManager(DataSource dataSource) {
        return new PgBouncerFamiliarTransactionManager(dataSource);
    }

    @Bean
    public DatabaseCredentials dbCredentials(
            @Value("${calendar.jdbc.hosts}") String groupsOrHosts,
            @Value("${calendar.jdbc.port}") int port,
            @Value("${calendar.jdbc.dbname}") String dbName,
            @Value("${calendar.jdbc.username}") String user,
            @Value("${calendar.jdbc.password:-}") String password,
            @Value("${calendar.jdbc.password.file:-}") String passwordFile) {
        if (password.isEmpty()) {
            password = new File2(passwordFile).asReaderTool().readLine();
        }
        return new DatabaseCredentials(Either.right(groupsOrHosts), port, dbName, user, password);
    }

    @Bean
    public DcAwareDynamicMasterSlaveDataSourceFactoryBean dataSource(
            DataSourceFactoryWithMetrics factoryWithMetrics,
            DatabaseCredentials credents, Conductor conductor,
            @Value("${calendar.jdbc.urlPrefix}") String urlPrefix,
            @Value("${calendar.jdbc.urlSuffix}") String urlSuffix) {
        val factory = new DcAwareDynamicMasterSlaveDataSourceFactoryBean();

        factory.setDataSourceFactory(factoryWithMetrics);
        factory.setDsUrlPrefix(urlPrefix);
        factory.setDsUrlSuffix(urlSuffix);
        factory.setUsername(credents.getUser());

        factory.setPassword(credents.getPassword());
        factory.setQuietMode(true);

        factory.setHostPortDbNames(
                credents.getGroupHosts().map(conductor::resolveHostsFromString).getOrElse(credents.getHost())
                        .map(host -> StringUtils.format("{}:{}/{}", host, credents.getPort(), credents.getDbName())));

        return factory;
    }

    @Bean
    public DataSourceFactoryWithMetrics dbcpDataSource(
            @Value("${jdbc.pool.max.total}") int maxTotal,
            @Value("${jdbc.pool.max.idle}") int maxIdle,
            @Value("${jdbc.pool.min.idle}") int minIdle,
            @Value("${jdbc.pool.max.wait.millis}") int maxWaitMillis,
            @Value("${jdbc.pool.ps.pooling}") boolean psPooling,
            @Value("${jdbc.pool.test.on.borrow}") boolean testOnBorrow,
            @Value("${jdbc.pool.test.when.idle}") boolean testWhenIdle,
            @Value("${jdbc.pool.ping.max.wait.secs}") int pingMaxWaitSecs,
            @Value("${jdbc.pool.time.between.eviction.runs.millis}") int timeBetweenEvictionRunsMillis,
            @Value("${jdbc.pool.num.tests.per.eviction.run}") int numTestsPerEvictionRun,
            @Value("${jdbc.ping.timeout.url.params}") String pingUrlParams,
            MeterRegistry registry) {
        val conf = new DbcpConfiguration();

        conf.setMaxTotal(maxTotal);
        conf.setMaxIdle(maxIdle);
        conf.setMinIdle(minIdle);
        conf.setTimeBetweenEvictionRunsMillis(60000);
        conf.setMaxWaitMillis(maxWaitMillis);
        conf.setPingMaxWaitSecs(pingMaxWaitSecs);
        conf.setPsPooling(psPooling);
        conf.setTestOnBorrow(testOnBorrow);
        conf.setTestWhileIdle(testWhenIdle);
        conf.setTimeBetweenEvictionRunsMillis(timeBetweenEvictionRunsMillis);
        conf.setNumTestsPerEvictionRun(numTestsPerEvictionRun);
        conf.setNoDelayDsUrlParamOverrides(UrlUtils.urlParametersToTuple2List(pingUrlParams).toMap());

        val dbcpFactory = new DataSourceFactoryWithMetrics(registry);
        dbcpFactory.setPoolConfiguration(conf);
        return dbcpFactory;
    }

}
