package ru.yandex.calendar.boot;

import java.util.Properties;

import lombok.val;
import org.apache.log4j.AsyncAppender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.context.ConfigurableApplicationContext;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.CalendarVersion;
import ru.yandex.calendar.frontend.ews.proxy.ExchangeRequestLogger;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.JsonLogPatternLayout;
import ru.yandex.calendar.logic.log.requests.RequestsLogger;
import ru.yandex.calendar.util.Environment;
import ru.yandex.calendar.util.conf.CalendarPropertiesLoader;
import ru.yandex.calendar.util.conf.Configuration;
import ru.yandex.misc.log.log4j.appender.ReopenOnHupFileAppender;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.main.MainSupport;
import ru.yandex.misc.property.PropertiesHolder;
import ru.yandex.misc.property.PropertiesUtils;
import ru.yandex.misc.version.AbstractVersion;

public abstract class CalendarMainSupport extends MainSupport {
    public static final ru.yandex.misc.log.mlf.Logger logger = LoggerFactory.getLogger(CalendarMainSupport.class);

    static {
        Configuration.initializeEnvironment(false);
    }

    @Override
    protected final void initializeCorba() {
    }

    @Override
    protected AbstractVersion getVersion() {
        return CalendarVersion.VERSION;
    }

    @Override
    protected void postProcessApplicationContextBeforeRefresh(ConfigurableApplicationContext configurableApplicationContext) {
        super.postProcessApplicationContextBeforeRefresh(configurableApplicationContext);
        configurableApplicationContext.addBeanFactoryPostProcessor(new BeanFactoryPostProcessor() {
            public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
                beanFactory.registerSingleton("calendarAppName", appName());
            }
        });
    }

    protected abstract CalendarAppName appName();

    private String logOutputDirectory() {
        return Environment.isDeveloperNotebook() ? "logs/" : "/var/log/calendar/";
    }

    @Override
    protected String logOutput() {
        return Environment.isDeveloperNotebook() ? "" : "/var/log/calendar/calendar." + appName().xmlName() + ".log";
    }

    @Override
    public Tuple2<Properties, String[]> loadProperties(String[] args) {
        String[] unparsedArgs = CalendarPropertiesLoader.loadForApp(args, appName());

        if (isDaemon()) {
            PropertiesUtils.dump(PropertiesHolder.properties(), logger);
        }

        return Tuple2.tuple(PropertiesHolder.properties(), unparsedArgs);
    }

    @Override
    public void initializeLogger() {
        super.initializeLogger();
        installJsonAppender();
        installJsonLogger("events", EventsLogger.JSONNAME);
        installJsonLogger("exchange", ExchangeRequestLogger.NAME);
        installJsonLogger("requests", RequestsLogger.NAME);
        if (Environment.isProductionOrPre()) {
            installErrorAppender();
        }
    }

    private void installErrorAppender() {
        val errorFileAppender = new ReopenOnHupFileAppender();
        errorFileAppender.setFile(logOutputDirectory() + "calendar." + appName().xmlName() + ".error.json.log");
        errorFileAppender.setLayout(new JsonLogPatternLayout(appName()));
        errorFileAppender.activateOptions();

        val asyncAppender = new AsyncAppender();
        asyncAppender.addAppender(errorFileAppender);
        asyncAppender.setThreshold(Level.WARN);

        Logger.getRootLogger().addAppender(asyncAppender);
    }

    private void installJsonLogger(String path, String name) {
        val fileAppender = new ReopenOnHupFileAppender();

        fileAppender.setFile(logOutputDirectory() + "calendar." + appName().xmlName() + "." + path + ".json.log");
        fileAppender.setLayout(new PatternLayout());
        fileAppender.activateOptions();

        val asyncAppender = new AsyncAppender();
        asyncAppender.addAppender(fileAppender);

        val logger = Logger.getLogger(name);
        logger.setAdditivity(false);
        logger.addAppender(asyncAppender);
    }

    private void installJsonAppender() {
        val fileAppender = new ReopenOnHupFileAppender();

        fileAppender.setFile(logOutputDirectory() + "calendar." + appName().xmlName() + ".json.log");
        fileAppender.setLayout(new JsonLogPatternLayout(appName()));

        fileAppender.activateOptions();

        val asyncAppender = new AsyncAppender();
        asyncAppender.addAppender(fileAppender);

        Logger.getRootLogger().addAppender(asyncAppender);
    }

    @Override
    protected final Class[] applicationContextPath() {
        return Cf.<Class>list()
                .plus1(CalendarInitContextConfiguration.class)
                .plus1(CalendarContextConfiguration.class)
                .plus(isDaemon()
                        ? Cf.list(CalendarDaemonContextConfiguration.class)
                        : Cf.list())
                .plus(applicationSpecificContextPath())
                .toArray(Class.class);
    }

    protected abstract Class[] applicationSpecificContextPath();

    @Override
    protected final void initializeAppSpecific() throws Exception {
        initializeCalendarAppSpecific();
    }

    protected void initializeCalendarAppSpecific() throws Exception {
    }

    @Override
    protected boolean isInitializeLogger() {
        return true;
    }

    @Override
    protected boolean isDaemon() {
        return true;
    }

} //~
