package ru.yandex.calendar.frontend.a3.bind;

import java.io.IOException;
import java.io.StringWriter;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.a3.interceptors.LoggingParameterBinder;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.Parameter;
import ru.yandex.commune.a3.action.parameter.ParameterDescriptor;
import ru.yandex.commune.a3.action.parameter.WebRequest;
import ru.yandex.commune.a3.action.parameter.bind.ParameterBinder;
import ru.yandex.commune.mail.ContentType;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.serialize.BenderJsonGeneratorWrapper;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author gutman
 */
public class JsonBinder implements ParameterBinder, LoggingParameterBinder {

    private final BenderMapper mapper;
    private final BenderMapper mapperForLogging;

    public JsonBinder(BenderMapper mapper) {
        this.mapper = mapper;
        this.mapperForLogging = mapper;
    }

    public JsonBinder(BenderConfiguration configuration, BenderConfiguration configurationForLogging) {
        this.mapper = new BenderMapper(configuration);
        this.mapperForLogging = new BenderMapper(configurationForLogging);
    }

    @Override
    public Object createAndBind(
            WebRequest webRequest, ActionInvocationContext invocationContext, ParameterDescriptor parameterDescriptor)
    {
        Class<?> parameterClass = parameterDescriptor.getParameterType().getActualType().asClass().getClazz();

        HttpServletRequestX request = webRequest.getHttpServletRequest();
        Option<ContentType> contentType = Option.ofNullable(request.getContentType()).map(ContentType.valueOfF());

        Option<String> json;

        if (contentType.isPresent() && contentType.get().getTypeSubtype().equals("application/x-www-form-urlencoded")) {
            json = request.getParameterO("data");
        } else {
            json = StringUtils.notEmptyO(request.getInputStreamX().readString());
        }
        return json.map(data -> mapper.parseJson(parameterClass, data)).getOrNull();
    }

    @Override
    public Option<String> getNameForLogging(Parameter parameter) {
        return Option.empty();
    }

    @Override
    public Option<String> getValueForLogging(Parameter parameter) {
        if (!parameter.getValue().isPresent()) {
            return Option.empty();
        }

        Object value = parameter.getValue().get();

        if (value instanceof Option) {
            if (!((Option) value).isPresent()) {
                return Option.empty();
            }
            value = ((Option) value).get();
        }

        try (StringWriter wr = new StringWriter();
             JsonGenerator gen = new JsonFactory().createGenerator(wr).useDefaultPrettyPrinter()) {
            BenderJsonWriter jw = new BenderJsonGeneratorWrapper(gen);
            mapperForLogging.serializeJson(value, jw);
            jw.close();
            return Option.of(wr.toString());

        } catch (IOException e) {
            throw IoUtils.translate(e);
        }
    }

    public BenderMapper getMapper() {
        return mapper;
    }
}
