package ru.yandex.calendar.frontend.a3.interceptors;

import java.util.Map;
import java.util.Optional;

import lombok.AllArgsConstructor;
import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.RemoteInfo;
import ru.yandex.calendar.frontend.AddrUtils;
import ru.yandex.calendar.frontend.HeaderUtils;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.tvm.TvmManager;
import ru.yandex.calendar.util.dates.Timeouts;
import ru.yandex.commune.a3.action.intercept.ActionInvocationInterceptor;
import ru.yandex.commune.a3.action.intercept.InvocationInterceptorOrders;
import ru.yandex.commune.a3.action.invoke.ActionInvocation;
import ru.yandex.commune.a3.action.invoke.CloneableActionInvocation;
import ru.yandex.commune.a3.security.UnauthenticatedException;
import ru.yandex.commune.a3.security.UnauthorizedException;
import ru.yandex.misc.log.mlf.ndc.Ndc;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.WithTlTimeoutInMillis;

@Slf4j
@AllArgsConstructor
public class CalendarRequestInterceptor implements ActionInvocationInterceptor {
    private final ActionSource actionSource;
    private final PassportAuthDomainsHolder domains;
    private final TvmManager tvmManager;

    @Override
    public Object intercept(ActionInvocation invocation) throws Exception {
        val actionName = invocation.getActionDescriptor().getName();

        val timeoutInMillis = invocation.getActionDescriptor().getAnnotationO(WithTlTimeoutInMillis.class);
        val timeoutByProduct = invocation.getActionDescriptor().getAnnotationO(WithTlTimeoutByProduct.class);

        val timeout = timeoutInMillis.map(WithTlTimeoutInMillis::value)
                .orElse(timeoutByProduct.map(t -> domains.containsYandexTeamRu() ? t.yt() : t.pub()))
                .map(Duration::millis).getOrElse(Timeouts.getForCommand());

        val ndcHandle = Ndc.push("a=" + actionName);
        val actionSource = getActionSource(invocation);
        val calendarRequestHandle = CalendarRequest.push(getRemoteInfo(invocation), actionSource,
            "action " + actionName, actionName);

        val tltHandle = ThreadLocalTimeout.push(timeout, "action " + actionName);

        val headerReader = HeaderUtils.getHeaderReader(invocation);
        val ip = AddrUtils.getIp(headerReader);
        try {
            val parameters = invocation.getWebRequest().getParameters();
            CalendarRequestInterceptorHelper.checkParameters(parameters);
            val uid = CalendarRequestInterceptorHelper.getUid(parameters);
            tvmManager.checkTickets(uid, headerReader, actionSource);
            return invocation.invoke();
        } catch (UnauthenticatedException | UnauthorizedException e) {
            log.error("Auth exception for IP {}", ip.orElse("UNKNOWN"), e);
            throw e;
        } finally {
            calendarRequestHandle.popSafely();
            ndcHandle.popSafely();
            tltHandle.popSafely();
        }
    }

    @Override
    public int getOrder() {
        return InvocationInterceptorOrders.COMMON_ATTRIBUTES_INTERCEPTOR_ORDER - 3;
    }

    private ActionSource getActionSource(ActionInvocation invocation) {
        if (invocation instanceof CloneableActionInvocation) {
            val action = ((CloneableActionInvocation) invocation).getAction();

            if (action instanceof ActionSourceAwareAction) {
                return ((ActionSourceAwareAction) action).getActionSource();
            }
        }
        val request = invocation.getWebRequest();

        if (actionSource == ActionSource.WEB && request.getParameter("__agent").containsTs("staff")) {
            return ActionSource.WEB_FOR_STAFF;
        }
        if (actionSource == ActionSource.WEB && request.getHeader("x-yandex-clienttype").isSome("MAYA")) {
            return ActionSource.WEB_MAYA;
        }
        return actionSource;
    }

    private RemoteInfo getRemoteInfo(ActionInvocation invocation) {
        return new RemoteInfo(
                invocation.getWebRequest().getHttpServletRequest().getXRealIp(),
                invocation.getWebRequest().getHeader("X-Yandexuid"),
                invocation.getWebRequest().getParameter("connectionId").firstO());
    }
}

@UtilityClass
class CalendarRequestInterceptorHelper {
    static void checkParameters(Map<String, ListF<String>> parameters) {
        val uid = getParameterValue(parameters, "uid");
        val actorUid = getParameterValue(parameters, "actorUid");
        val targetUid = getParameterValue(parameters, "targetUid");

        if (uid.isPresent() && (actorUid.isPresent() || targetUid.isPresent())) {
            throw new IllegalArgumentException("Parameter uid cannot be passed along with parameter actorUid or targetUid");
        }
        if (targetUid.isPresent() && actorUid.isEmpty()) {
            throw new IllegalArgumentException("Cannot accept targetUid parameter without actorUid parameter");
        }
    }

    static Optional<Long> getUid(Map<String, ListF<String>> parameters) {
        val uid = getParameterValue(parameters, "uid");
        val actorUid = getParameterValue(parameters, "actorUid");
        return uid.or(() -> actorUid).map(CalendarRequestInterceptorHelper::parseLong);
    }

    private static long parseLong(String uid) {
        try {
            return Long.parseLong(uid);
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException("Parameter uid could not be parsed appropriately");
        }
    }

    static Optional<String> getParameterValue(Map<String, ListF<String>> parameters, String key) {
        return Optional.ofNullable(parameters.get(key))
                .map(StreamEx::of)
                .flatMap(StreamEx::findFirst);
    }
}
