package ru.yandex.calendar.frontend.api;

import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.domain.DomainManager;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.LayerIdPredicate;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.model.Completion;
import ru.yandex.calendar.logic.event.model.EventUserData;
import ru.yandex.calendar.logic.event.model.Priority;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.notification.NotificationRoutines;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author gutman
 */
public class ApiManager {
    private static final Logger logger = LoggerFactory.getLogger(ApiManager.class);

    @Autowired
    DomainManager domainManager;
    @Autowired
    private EventRoutines eventRoutines;

    public EventUserData eventUserData(PassportUid uid, ListF<String> notificationChannels, String notifyInMins) {
        ListF<Duration> offsets = NotificationRoutines.splitOffsets(notifyInMins).map(AuxDateTime.minutesToDurationF());
        ListF<Notification> notifications = Cf.arrayList();

        if (notificationChannels.containsTs("email")) {
            notifications.addAll(offsets.map(Notification::email));
        }
        if (notificationChannels.containsTs("sms")) {
            notifications.addAll(offsets.map(Notification::sms));
        }
        if (notificationChannels.containsTs("smsOrElseEmail")) {
            if (domainManager.getPhoneNumber(uid).isPresent()) {
                notifications.addAll(offsets.map(Notification::sms));
            } else {
                notifications.addAll(offsets.map(Notification::email));
            }
        }

        EventUser eventUser = new EventUser();
        eventUser.setAvailability(Availability.MAYBE);
        eventUser.setCompletion(Completion.NOT_APPLICABLE);
        eventUser.setPriority(Priority.NORMAL);
        eventUser.setIsAttendee(true);
        eventUser.setIsOptional(false);

        return new EventUserData(eventUser, notifications);
    }

    public EventAndNeighbourEvents getEventAndNeighbourEventsByEvent(
            final Event event, PassportUid uid, DateTimeZone tz)
    {
        Instant startOfDay = event.getStartTs().toDateTime(tz).withTimeAtStartOfDay().toInstant();
        Instant endOfDay = event.getStartTs().toDateTime(tz).plusDays(1).withTimeAtStartOfDay().toInstant();

        ListF<EventInstanceInfo> thisDayEvents = eventRoutines.getSortedInstancesIMayView(
                Option.of(uid), startOfDay, Option.of(endOfDay),
                LayerIdPredicate.allForUser(uid, true), ActionSource.INTERNAL_API).sortedBy(EventInstanceInfo.startF());

        ListF<EventInstanceInfo> thisDayEventsExceptThis = thisDayEvents
                .filter(EventInstanceInfo.eventIdF().andThenEquals(event.getId()).notF());
        if (thisDayEvents.size() == thisDayEventsExceptThis.size()) {
            logger.warn("No requested event among this day events!");
        }

        ListF<EventInstanceInfo> eventsBeforeEvent = thisDayEventsExceptThis.takeWhile(EventInstanceInfo.startF().andThen(new Function1B<Instant>() {
            public boolean apply(Instant eventStart) {
                return eventStart.isBefore(event.getStartTs());
            }
        }));
        Option<EventInstanceInfo> previousEvent = eventsBeforeEvent.lastO();
        Option<EventInstanceInfo> nextEvent = thisDayEventsExceptThis.drop(eventsBeforeEvent.size()).firstO();

        return new EventAndNeighbourEvents(
                event, previousEvent.map(EventInstanceInfo.getEventF()), nextEvent.map(EventInstanceInfo.getEventF()));
    }
}
