package ru.yandex.calendar.frontend.api;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.frontend.a3.error.WithHttpErrorCodes;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.archive.ArchiveManager;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.change.EventChangeLogEvents;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.commune.a3.action.http.HttpStatusCodeSource;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.security.SecurityErrorNames;
import ru.yandex.commune.a3.security.UnauthorizedException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.io.http.HttpStatus;

@WithHttpErrorCodes
public class DeleteSpamEventAction extends XmlOrJsonApiActionSupport {

    private static final DynamicProperty<ListF<Long>> permittedTvmClients =
            new DynamicProperty<>("tvm.actions.deleteSpamEvent.permitted", Cf.list());

    @RequestParam
    private long eventId;

    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private EventsLogger eventsLogger;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private ArchiveManager archiveManager;

    @Override
    protected XmlOrJsonWritable doExecute() {
        if (getActionInfo().getTvmId().stream().noneMatch(permittedTvmClients.get()::containsTs)) {
            throw new UnauthorizedException(SecurityErrorNames.NOT_AUTHORIZED, "Not permitted to execute this action");
        }

        Option<MainEvent> mainEvent = mainEventDao.findMainEventsByEventIds(Cf.list(eventId)).singleO();

        if (mainEvent.isEmpty()) {
            throw new EventNotFoundException(eventId);
        }
        String externalId = mainEvent.get().getExternalId();

        return lockTransactionManager.lockAndDoInTransaction(LockResource.event(externalId), () -> {
            ActionInfo actionInfo = getActionInfo().withNow(Instant.now());

            ListF<EventAndRepetition> events = eventDbManager
                    .getEventsAndRepetitionsByMainEventId(mainEvent.get().getId())
                    .sortedBy(e -> e.getEvent().getRecurrenceId().orElse(new Instant(0)));

            EventAndRepetition master = events.find(e -> e.getEvent().getRepetitionId().isPresent())
                    .orElse(() -> events.find(e -> e.getEvent().getId() == eventId))
                    .orElseGet(events::first);
            ListF<Long> eventIds = events.map(EventAndRepetition::getEventId);

            ListF<EventLayer> eventLayers = eventLayerDao.findLayersIdsByEventIds(eventIds).map(layerId -> {
                EventLayer el = new EventLayer();
                el.setEventId(master.getEventId());
                el.setLayerId(layerId);
                return el;
            });
            Event dataToArchive = master.getEvent().copy();
            dataToArchive.setName("<$0>");
            dataToArchive.setRecurrenceId(Option.empty());

            archiveManager.storeDeletedEventLayers(eventLayers, actionInfo);
            archiveManager.storeDeletedEvents(Tuple2List.fromPairs(dataToArchive, externalId), actionInfo);
            eventDbManager.deleteEventsByIds(eventIds, true, actionInfo);

            events.forEach(event -> eventsLogger.log(EventChangeLogEvents.deletedSpam(externalId, event), actionInfo));

            return XmlOrJsonWritable.empty();
        });
    }

    private static class EventNotFoundException extends RuntimeException implements HttpStatusCodeSource {

        public EventNotFoundException(long eventId) {
            super("No event found by id " + eventId);
        }

        @Override
        public int getHttpStatusCode() {
            return HttpStatus.SC_404_NOT_FOUND;
        }

        @Override
        public Throwable fillInStackTrace() {
            return this;
        }
    }
}
