package ru.yandex.calendar.frontend.api.event;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.a3.converters.ConverterToInstantIsoDateTime;
import ru.yandex.calendar.frontend.api.ApiManager;
import ru.yandex.calendar.frontend.api.XmlApiActionSupport;
import ru.yandex.calendar.frontend.api.XmlWritable;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.CreateInfo;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.web.EventWebManager;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.sending.so.SoChecker;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author gutman
 */
public class CreateEventAction extends XmlApiActionSupport {
    private static final Logger logger = LoggerFactory.getLogger(CreateEventAction.class);

    @Autowired
    private UserManager userManager;
    @Autowired
    private ApiManager apiManager;
    @Autowired
    private EventWebManager eventWebManager;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    protected EventRoutines eventRoutines;
    @Autowired
    private SoChecker soChecker;

    @RequestParam
    private String name;
    @RequestParam(customConverter = ConverterToInstantIsoDateTime.class)
    private Instant start;
    @RequestParam(customConverter = ConverterToInstantIsoDateTime.class)
    private Instant end;
    @RequestParam
    private Email email;

    @RequestParam
    private Option<String> service = Option.empty();

    @RequestListParam(required = false)
    private ListF<String> notification;
    @RequestParam(required = false)
    private int advanceMins = 15;

    @RequestParam("place")
    private Option<String> placeO = Option.empty();

    @Override
    protected XmlWritable doExecute() {
        logger.debug("User " + email + " creates event " + name);

        Validate.V.isTrue("start must be before or equal to end, start: " + start + ", end: " + end,
                !start.isAfter(end));

        PassportUid uid = userManager.getUidByEmail(email).getOrThrow("unknown user " + email);

        EventData eventData = eventData(uid);

        ActionInfo actionInfo = getActionInfo();

        CreateInfo info;

        if (service.isPresent()) {
            Validate.equals("tv", service.get());
            info = eventRoutines.createServiceEvent(uid, PassportSid.TV, eventData,
                    NotificationsData.createFromWeb(eventData.getEventUserData().getNotifications()), actionInfo);
        } else {
            info = eventWebManager.createUserEvent(uid, eventData, InvitationProcessingMode.SAVE_ONLY, actionInfo);
        }

        return XmlWritable.textElement("event-id", info.getEvent().getId());
    }

    private EventData eventData(PassportUid uid) {
        EventData eventData = new EventData();
        eventData.getEvent().setStartTs(start);
        eventData.getEvent().setEndTs(end);
        eventData.getEvent().setName(name);
        eventData.getEvent().setIsAllDay(false);
        eventData.setTimeZone(dateTimeManager.getTimeZoneForUid(uid));

        if (placeO.isPresent()) {
            eventData.getEvent().setLocation(placeO.get());
        }
        eventData.setEventUserData(apiManager.eventUserData(uid, notification, "-" + advanceMins));
        return eventData;
    }

    @Deprecated
    @RequestParam(required = false)
    // for compatibility with moikrug
    public void setLogin(String login) {
        this.email = new Email(login);
    }
}
