package ru.yandex.calendar.frontend.api.event;

import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.api.EventAndNeighbourEvents;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.xmlorjson.XmlOrJsonWriter;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author gutman
 */
public class EventSerializer {

    public static void serializeDecisions(XmlOrJsonWriter xw, EventUser eventUser) {
        final ListF<Decision> possibleDecisions;
        if (eventUser.getDecision() == Decision.UNDECIDED) {
            possibleDecisions = Decision.R.valuesList();
        } else {
            possibleDecisions = Cf.list(Decision.YES, Decision.MAYBE, Decision.NO); // can't go back to undecided
        }
        xw.startArray("decisions");
        for (Decision d : possibleDecisions) {
            xw.startObject("decision");
            xw.addTextField("value", d.name());
            xw.addTextField("is-current", Boolean.toString(eventUser.getDecision() == d));
            xw.endObject();
        }
        xw.endArray();
    }

    public static void serializeEventFields(XmlOrJsonWriter w, Event event, Option<DateTimeZone> timezone) {
        serializeEventFields(w, event, new InstantInterval(event.getStartTs(), event.getEndTs()), timezone);
    }

    public static void serializeEventFields(
            XmlOrJsonWriter w, Event event, InstantInterval interval, Option<DateTimeZone> timezone)
    {
        w.addTextField("id", event.getId());
        w.addTextField("name", event.getName());
        w.addTextField("description", event.getDescription());
        w.addTextField("location", event.getLocation());

        if (timezone.isPresent()) {
            DateTimeFormatter formatter = ISODateTimeFormat.dateTime().withZone(timezone.get());
            w.addTextField("start", formatter.print(interval.getStart()));
            w.addTextField("end", formatter.print(interval.getEnd()));
        } else {
            w.addNumberField("start", interval.getStart().getMillis());
            w.addNumberField("end", interval.getEnd().getMillis());
        }
    }

    public static void serializeEventAndNeighbourEvents(XmlOrJsonWriter w, DateTimeZone timezone, EventAndNeighbourEvents eventAndNeighbours) {
        w.startObject("event");

        serializeEventFields(w, eventAndNeighbours.getEvent(), Option.of(timezone));

        w.endObject();

        if (eventAndNeighbours.getPreviousEvent().isPresent()) {
            w.startObject("previous-event");
            serializeEventFields(w, eventAndNeighbours.getPreviousEvent().get(), Option.of(timezone));
            w.endObject();
        }
        if (eventAndNeighbours.getNextEvent().isPresent()) {
            w.startObject("next-event");
            serializeEventFields(w, eventAndNeighbours.getNextEvent().get(), Option.of(timezone));
            w.endObject();
        }
    }

    public static void serializeParticipants(XmlOrJsonWriter w, ListF<ParticipantInfo> participants) {
        w.startArray("participants");

        for (ParticipantInfo participant : participants) {
            w.startObject("participant");

            Option<String> login = participant.isYandexUser()
                    ? ((YandexUserParticipantInfo) participant).getLogin() : Option.empty();

            w.addTextField("name", participant.getName());
            w.addTextField("email", Emails.getUnicoded(participant.getEmail()));
            w.addTextField("login", login.getOrElse(""));

            w.addTextField("decision", participant.getDecision().toDbValue());
            w.addTextField("is-organizer", String.valueOf(participant.isOrganizer()));
            w.addTextField("is-resource", String.valueOf(participant.isResource()));

            w.endObject();
        }
        w.endArray();
    }

}
