package ru.yandex.calendar.frontend.api.event;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.api.XmlOrJsonWritable;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventGetProps;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventLoadLimits;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventUserWithRelations;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.LayerIdPredicate;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.calendar.util.xmlorjson.XmlOrJsonWriter;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.MoscowTime;

/**
 * @author dbrylev
 */
@Action({@Action.Alias("getNowEventsOnLayer")})
@WithMasterSlavePolicy(MasterSlavePolicy.R_MS)
public class GetEventsOnLayer extends ActionWithIntervalParams {

    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private LayerRoutines layerRoutines;

    @RequestParam
    private Option<String> layerPrivateToken;
    @RequestParam
    private Option<Long> layerId;

    @Override
    protected XmlOrJsonWritable doExecute() {
        Validate.isTrue(layerId.isPresent() || layerPrivateToken.isPresent(), "layer id or token must be specified");

        long layerId = this.layerId.getOrElse(() -> layerRoutines.getByPrivateToken(layerPrivateToken.get()).getId());
        InstantInterval interval = new InstantInterval(getFrom(), getTo());

        boolean tokenAccessed = !this.layerId.isPresent();

        EventGetProps egp = EventGetProps.any();
        LayerIdPredicate layerIds = LayerIdPredicate.list(Cf.list(layerId), false);
        EventLoadLimits limits = EventLoadLimits.intersectsInterval(interval);

        final ListF<EventInstanceInfo> events = eventRoutines.getSortedInstancesOnLayer(
                Option.<PassportUid>empty(), egp, layerIds, limits, ActionSource.INTERNAL_API);

        return new XmlOrJsonWritable() {
            public void write(XmlOrJsonWriter w) {
                w.startArray("events");
                for (EventInstanceInfo event : events) {
                    if (!tokenAccessed && !event.getMayView()) continue;

                    w.startObject("event");

                    EventSerializer.serializeEventFields(w, event.getEvent(),
                            event.getEventInterval().toInstantInterval(MoscowTime.TZ), Option.of(MoscowTime.TZ));

                    EventSerializer.serializeParticipants(w,
                            getParticipantsOrElseCreatorAsOrganizer(event.getEventWithRelations()));

                    w.endObject();
                }
                w.endArray();
            }
        };
    }

    private ListF<ParticipantInfo> getParticipantsOrElseCreatorAsOrganizer(EventWithRelations event) {
        if (event.getParticipants().isMeetingOrInconsistent()) {
            return event.getParticipants().getParticipantsSafeWithInconsistent();

        } else {
            PassportUid creatorUid = event.getEvent().getCreatorUid();
            Option<EventUserWithRelations> creatorEu = event.findUserEventUserWithRelations(creatorUid);

            return creatorEu.map(new Function<EventUserWithRelations, ParticipantInfo>() {
                public ParticipantInfo apply(EventUserWithRelations eu) {
                    EventUser eventUser = eu.getEventUser().copy();
                    eventUser.setIsOrganizer(true);
                    eventUser.setIsAttendee(true);
                    eventUser.setIsOptional(false);

                    return new YandexUserParticipantInfo(eventUser, eu.getSettings());
                }
            });
        }
    }
}
